package pro.mypvit.exceptions;

import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import lombok.Data;

import java.util.Map;
import java.util.Set;

/**
 * Represents an error response returned by the API.
 *
 * <p>This class captures details about errors returned by the API, such as the status code,
 * error message, and additional metadata. It also provides functionality for JSON deserialization.</p>
 *
 * Key Features:
 * <ul>
 *     <li>Encapsulates error details including date, status, and messages.</li>
 *     <li>Supports deserialization from JSON using a predefined {@link Gson} configuration.</li>
 *     <li>Includes a default constructor for easy instantiation.</li>
 * </ul>
 *
 * Usage Example:
 * <pre>
 * String jsonResponse = "{"
 *     + "\"date\": \"2025-01-01T10:00:00Z\","
 *     + "\"status_code\": \"400\","
 *     + "\"error\": \"Bad Request\","
 *     + "\"message\": \"Invalid request data\","
 *     + "\"path\": \"/api/transaction\","
 *     + "\"messages\": [\"Field 'amount' is required\"],"
 *     + "\"errors\": {\"amount\": \"Must be greater than 0\"}"
 *     + "}";
 *
 * ApiError error = ApiError.fromJson(jsonResponse);
 * System.out.println("Error: " + error.getMessage());
 * </pre>
 */
@Data
public class ApiError {
    /**
     * The timestamp when the error occurred.
     */
    private String date;

    /**
     * The HTTP status code associated with the error.
     */
    private String statusCode;

    /**
     * The error type or category.
     */
    private String error;

    /**
     * The main error message describing the issue.
     */
    private String message;

    /**
     * The API path where the error occurred.
     */
    private String path;

    /**
     * A set of additional error messages, often used for validation errors.
     */
    private Set<String> messages;

    /**
     * A map of specific field errors, with the field name as the key and the error as the value.
     */
    private Map<String, String> errors;

    /**
     * A pre-configured {@link Gson} instance for JSON deserialization.
     */
    private static final Gson gson = new GsonBuilder()
            .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
            .create();

    /**
     * Default constructor for creating an empty {@code ApiError} instance.
     */
    public ApiError() {
        // Default constructor for easy instantiation
    }

    /**
     * Converts a JSON string into an {@code ApiError} instance.
     *
     * @param json The JSON string to deserialize.
     * @return An {@code ApiError} instance populated with data from the JSON string.
     */
    public static ApiError fromJson(String json) {
        return gson.fromJson(json, ApiError.class);
    }
}