package pro.mypvit.exceptions;

import okhttp3.Response;

import java.io.IOException;

/**
 * Utility class for handling API errors in the MyPVit SDK.
 *
 * <p>This class provides functionality to process HTTP responses, identify API errors, and
 * throw custom exceptions with detailed information. It also includes logging for debugging purposes.</p>
 */
public class ExceptionUtils {

    /**
     * Handles API errors by inspecting the HTTP response and throwing a {@link MyPVitException}
     * if an error is detected.
     *
     * <p>If the response indicates a failure (non-2xx status), this method extracts the error
     * details from the response body, converts it into an {@link ApiError} object, and throws
     * a custom {@link MyPVitException}. It also logs the error details to the console.</p>
     *
     * @param response The HTTP response to check.
     * @throws MyPVitException If the response indicates an API error.
     * @throws RuntimeException If an I/O error occurs while reading the response body.
     */
    public static void handleApiError(Response response) {
        try {
            if (!response.isSuccessful()) {
                // Extract error details from the response body
                String errorBody = response.body() != null ? response.body().string() : null;
                ApiError apiError = errorBody != null ? ApiError.fromJson(errorBody) : new ApiError();

                // Create a custom exception with detailed information
                MyPVitException exception = new MyPVitException(
                        apiError.getMessage() != null ? apiError.getMessage() : "An unexpected error occurred",
                        apiError.getDate(),
                        apiError.getStatusCode(),
                        apiError.getError(),
                        apiError.getPath(),
                        apiError.getMessages(),
                        apiError.getErrors()
                );

                // Log the error and throw the exception
                logError(exception);
                throw exception;
            }
        } catch (IOException e) {
            System.err.println("Failed to process API error response: " + e.getMessage());
            throw new RuntimeException(e);
        }
    }

    /**
     * Logs the details of a {@link MyPVitException} to the console.
     *
     * <p>This method prints the exception's message to {@code System.err} for debugging purposes.
     * It is primarily used internally by {@link #handleApiError(Response)}.</p>
     *
     * @param exception The exception to log.
     */
    private static void logError(MyPVitException exception) {
        // Print the error message to the standard error output
        System.err.println(exception.getMessage());
    }

    /**
     * Default constructor
     */
    public ExceptionUtils() {}
}

