package pro.mypvit.exceptions;

import lombok.*;

import java.util.Map;
import java.util.Set;

/**
 * Custom exception class for handling API errors in the MyPVit SDK.
 *
 *
 *     This exception encapsulates detailed error information returned by the API,
 *     such as the status code, error type, and specific field errors. It also provides
 *     a custom implementation of {@link #getMessage()} to construct a detailed error message
 *     for logging and debugging purposes.
 *
 *
 * Key Features:
 * <ul>
 *     <li>Extends {@link RuntimeException}, allowing unchecked exceptions to be thrown.</li>
 *     <li>Includes additional fields to capture specific error details.</li>
 *     <li>Provides a custom {@link #getMessage()} implementation for detailed error reporting.</li>
 * </ul>
 */
@EqualsAndHashCode(callSuper = true)
@Getter
@Setter
@ToString
@NoArgsConstructor
public class MyPVitException extends RuntimeException {

    /**
     * The timestamp when the error occurred.
     */
    private String date;

    /**
     * The HTTP status code associated with the error.
     */
    private String statusCode;

    /**
     * The type or category of the error.
     */
    private String error;

    /**
     * The API path where the error occurred.
     */
    private String path;

    /**
     * A set of constraint violation messages, often used for validation errors.
     */
    private Set<String> messages;

    /**
     * A map of specific field errors, with field names as keys and error messages as values.
     */
    private Map<String, String> errors;

    /**
     * Constructs a new {@code MyPVitException} with the specified error details.
     *
     * @param message    The main error message.
     * @param date       The timestamp of the error.
     * @param statusCode The HTTP status code of the error.
     * @param error      The type or category of the error.
     * @param path       The API path where the error occurred.
     * @param messages   A set of constraint violation messages.
     * @param errors     A map of field-specific error messages.
     */
    public MyPVitException(String message, String date, String statusCode, String error, String path, Set<String> messages, Map<String, String> errors) {
        super(message);
        this.date = date;
        this.statusCode = statusCode;
        this.error = error;
        this.path = path;
        this.messages = messages;
        this.errors = errors;
    }

    /**
     * Constructs a detailed error message for logging and debugging purposes.
     *
     * The custom error message includes:
     * <ul>
     *     <li>The main error message.</li>
     *     <li>Constraint violation messages, if any.</li>
     *     <li>Field-specific error details.</li>
     *     <li>General error information, such as status code and API path.</li>
     * </ul>
     *
     *
     * @return A string representation of the error with detailed information.
     */
    @Override
    public String getMessage() {
        StringBuilder errorMessage = new StringBuilder();

        if (super.getMessage() != null) {
            errorMessage.append("Message: ").append(super.getMessage()).append("\n");
        }

        if (messages != null && !messages.isEmpty()) {
            errorMessage.append("Constraint Violations: ").append(messages).append("\n");
        }

        if (errors != null && !errors.isEmpty()) {
            errorMessage.append("Field Errors: ").append(errors).append("\n");
        }

        if (error != null) {
            errorMessage.append("Error: ").append(error).append("\n");
        }
        if (statusCode != null) {
            errorMessage.append("Status Code: ").append(statusCode).append("\n");
        }
        if (path != null) {
            errorMessage.append("Path: ").append(path).append("\n");
        }
        if (date != null) {
            errorMessage.append("Date: ").append(date).append("\n");
        }

        return errorMessage.toString();
    }
}