package pro.mypvit.requests;

import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import lombok.AllArgsConstructor;
import lombok.experimental.SuperBuilder;
/**
 * Abstract base class for all request objects in the MyPVit SDK.
 * This class provides a shared functionality to convert request objects to JSON format,
 * ensuring consistent serialization with a specified field naming policy.
 *
 * Features:
 * <ul>
 *     <li>Uses {@link SuperBuilder} for enabling builder pattern in subclasses.</li>
 *     <li>Provides a utility method for converting the object to JSON using Gson.</li>
 * </ul>
 *
 * Serialization Policy:
 * The field naming policy for JSON serialization is set to {@link FieldNamingPolicy#LOWER_CASE_WITH_UNDERSCORES},
 *     ensuring that field names in JSON match the snake_case naming convention used in the API.
 */
@SuperBuilder
public abstract class AbstractRequest {
    private static final Gson gson = new GsonBuilder()
            .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
            .create();

    /**
     * Converts the current object to its JSON representation.
     * This method uses a pre-configured {@link Gson} instance to serialize the object,
     * ensuring that the JSON follows the field naming policy.
     *
     * @return A JSON string representation of the object.
     */
    public String toJson() {
        return gson.toJson(this);
    }

    /**
     * Default constructor
     */
    public AbstractRequest() {
    }
}
