package pro.mypvit.requests;

import lombok.*;
import lombok.experimental.SuperBuilder;

/**
 * Base class for transaction requests in the MyPVit SDK.
 * This class provides common fields for all transaction requests, such as agent information,
 * amount, and product details.
 *
 * <p>It is designed to be extended by specific classes for different types of transactions.</p>
 *
 * Annotations Used:
 * <ul>
 *     <li>{@link Data} : Automatically generates getters, setters, and other utility methods.</li>
 *     <li>{@link SuperBuilder} : Enables builder pattern usage with inheritance.</li>
 *     <li>{@link NoArgsConstructor} : Creates a no-arguments constructor.</li>
 *     <li>{@link AllArgsConstructor} : Creates a constructor with all arguments.</li>
 *     <li>{@link EqualsAndHashCode} : Generates equals() and hashCode() methods with inheritance support.</li>
 * </ul>
 */
@Data
@SuperBuilder
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class BaseTransactionRequest extends AbstractRequest {
    /**
     * The identifier of the agent responsible for the transaction.
     */
    private String agent;

    /**
     * The amount involved in the transaction.
     */
    private Double amount;

    /**
     * The product associated with the transaction.
     */
    private String product;

    /**
     * A unique reference for the transaction.
     */
    private String reference;

    /**
     * The customer account number involved in the transaction.
     */
    private String customerAccountNumber;

    /**
     * The service used for the transaction (e.g., RESTFUL, RESTLINK, VISA_MASTERCARD).
     */
    private String service;

    /**
     * The callback URL code for this transaction.
     */
    private String callbackUrlCode;

    /**
     * The operation account code for the merchant involved in the transaction (e.g, ACC_6718284F467E6).
     */
    private String merchantOperationAccountCode;

    /**
     * The type of the transaction (e.g., PAYMENT, GIVE_CHANGE).
     */
    private String transactionType;

    /**
     * The owner of the charge for this transaction (e.g., CUSTOMER, MERCHANT).
     */
    private String ownerCharge;

    /**
     * The owner responsible for the operator charge (e.g., CUSTOMER, MERCHANT. default CUSTOMER).
     */
    private String ownerChargeOperator;

    /**
     * Additional information or metadata associated with the transaction.
     */
    private String freeInfo;

    /**
     * Default constructor
     */
    public BaseTransactionRequest() {}
}
