package pro.mypvit.requests;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Represents a request for initiating a transaction using the LINK API in the MyPVit SDK.
 *
 * <p>This class extends {@link BaseTransactionRequest} to inherit common transaction fields
 * while adding fields specific to LINK transactions, such as redirection URLs for success
 * and failure scenarios.</p>
 *
 * Key Features:
 * <ul>
 *     <li>Inherits all fields and functionalities of {@link BaseTransactionRequest}.</li>
 *     <li>Adds fields for managing redirection URLs upon transaction success or failure.</li>
 *     <li>Supports builder pattern for object creation through {@link SuperBuilder}.</li>
 * </ul>
 *
 * Usage:
 * This class is specifically designed for requests sent to the LINK API endpoint of MyPVit.
 * Use the builder pattern to create instances of this class and populate the required fields.
 *
 * Example:
 * <pre>
 * TransactionLinkRequest request = TransactionLinkRequest.builder()
 *     .agent("AGENT-1")
 *     .amount(100.0)
 *     .product("PRODUCT_TEST")
 *     .reference("REF123456")
 *     .customerAccountNumber("123456789")
 *     .service("RESTLINK")
 *     .callbackUrlCode("GIGMY")
 *     .merchantOperationAccountCode("ACC_6718284F467E6")
 *     .transactionType("PAYMENT")
 *     .ownerCharge("CUSTOMER")
 *     .ownerChargeOperator("MERCHANT")
 *     .freeInfo("Some additional info")
 *     .failedRedirectionUrlCode("CBFOU")
 *     .successRedirectionUrlCode("6YRHJ")
 *     .build();
 *
 * String json = request.toJson();
 * System.out.println(json);
 * </pre>
 */
@Data
@SuperBuilder
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class TransactionLinkRequest extends BaseTransactionRequest{
    /**
     * The URL code to redirect to if the transaction fails.
     */
    private String failedRedirectionUrlCode;

    /**
     * The URL code to redirect to if the transaction succeeds.
     */
    private String successRedirectionUrlCode;

    /**
     * Default constructor
     */
    public TransactionLinkRequest() {}
}
