package pro.mypvit.responses;

import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
/**
 * Abstract base class for response objects in the MyPVit SDK.
 *
 * <p>This class provides a shared functionality to deserialize JSON strings into
 * response objects. It ensures consistent deserialization by using a pre-configured
 * {@link Gson} instance with a specific field naming policy.</p>
 *
 * Key Features:
 * <ul>
 *     <li>Generic deserialization method for converting JSON strings into specific response types.</li>
 *     <li>Uses {@link FieldNamingPolicy#LOWER_CASE_WITH_UNDERSCORES} to handle API responses
 *     that use snake_case field naming.</li>
 *     <li>Designed to be extended by specific response classes in the SDK.</li>
 * </ul>
 *
 * Usage:
 * Subclasses should extend this class to inherit the `fromJson` method for easy deserialization.
 */
public abstract class AbstractResponse {
    /**
     * Pre-configured {@link Gson} instance for JSON deserialization.
     */
    private static final Gson gson = new GsonBuilder()
            .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
            .create();

    /**
     * Converts a JSON string into an instance of a specified subclass of {@code AbstractResponse}.
     *
     * <p>This method uses a generic type to allow deserialization into any class
     * that extends {@code AbstractResponse}.</p>
     *
     * @param json The JSON string to deserialize.
     * @param clazz The class type of the subclass to deserialize into.
     * @param <T> The type of the subclass.
     * @return An instance of the specified subclass populated with data from the JSON string.
     * @throws com.google.gson.JsonSyntaxException if the JSON string is not a valid representation for the specified class.
     */
    public static <T extends AbstractResponse> T fromJson(String json, Class<T> clazz) {
        return gson.fromJson(json, clazz);
    }

    /**
     * Default constructor
     */
    public AbstractResponse() {}
}
