package pro.mypvit.responses;

import lombok.Data;
import lombok.EqualsAndHashCode;

/**
 * Represents the response for a transaction status inquiry in the MyPVit SDK.
 *
 * <p>This class extends {@link AbstractResponse} to inherit functionality for JSON deserialization
 * and provides specific fields related to transaction status details.</p>
 *
 * Key Features:
 * <ul>
 *     <li>Inherits JSON deserialization capabilities from {@link AbstractResponse}.</li>
 *     <li>Includes fields to describe the status, amount, fees, and other transaction details.</li>
 *     <li>Supports automatic generation of getters, setters, equals, and hashCode methods through Lombok.</li>
 * </ul>
 *
 * Usage:
 * Use the `fromJson` method provided by {@link AbstractResponse} to deserialize a JSON response into
 * an instance of this class.
 *
 * Example:
 * <pre>
 * String jsonResponse = "{"
 *     + "\"date\": \"2025-01-02T12:34:56Z\","
 *     + "\"status\": \"SUCCESS\","
 *     + "\"amount\": 150.0,"
 *     + "\"fees\": 2.5,"
 *     + "\"operator\": \"MERCHANT\","
 *     + "\"merchant_reference_id\": \"REF123456\","
 *     + "\"customer_account_number\": \"123456789\","
 *     + "\"merchant_operation_account_code\": \"MERCHANT_ACC\""
 *     + "}";
 *
 * StatusResponse response = AbstractResponse.fromJson(jsonResponse, StatusResponse.class);
 * System.out.println("Date: " + response.getDate());
 * System.out.println("Status: " + response.getStatus());
 * System.out.println("Amount: " + response.getAmount());
 * System.out.println("Fees: " + response.getFees());
 * </pre>
 */
@Data
@EqualsAndHashCode(callSuper = true)
public class StatusResponse extends AbstractResponse {
    /**
     * The date and time of the transaction status.
     */
    private String date;

    /**
     * The current status of the transaction (e.g., SUCCESS, PENDING, FAILED, AMBIGUOUS).
     */
    private String status;

    /**
     * The transaction amount.
     */
    private Double amount;

    /**
     * The fees applied to the transaction.
     */
    private Double fees;

    /**
     * The operator involved in the transaction (e.g., CUSTOMER, MERCHANT).
     */
    private String operator;

    /**
     * The merchant's reference ID for the transaction.
     */
    private String merchantReferenceId;

    /**
     * The customer's account number associated with the transaction.
     */
    private String customerAccountNumber;

    /**
     * The merchant's operation account code.
     */
    private String merchantOperationAccountCode;

    /**
     * Default constructor
     */
    public StatusResponse() {}
}
