package pro.mypvit.responses;

import lombok.Data;
import lombok.EqualsAndHashCode;

/**
 * Represents the response for a LINK transaction request in the MyPVit SDK.
 *
 * <p>This class extends {@link AbstractResponse} to inherit functionality for JSON deserialization
 * and provides specific fields related to LINK transaction results.</p>
 *
 * Key Features:
 * <ul>
 *     <li>Inherits JSON deserialization capabilities from {@link AbstractResponse}.</li>
 *     <li>Includes fields for the transaction status, merchant reference, and URL details.</li>
 *     <li>Supports automatic generation of getters, setters, equals, and hashCode methods through Lombok.</li>
 * </ul>
 *
 * Usage:
 * Use the `fromJson` method provided by {@link AbstractResponse} to deserialize a JSON response into
 * an instance of this class.
 *
 * Example:
 * <pre>
 * String jsonResponse = "{"
 *     + "\"status\": \"SUCCESS\","
 *     + "\"status_code\": \"200\","
 *     + "\"merchant_reference_id\": \"REF123456\","
 *     + "\"url\": \"https://api.mypvit.pro/XXXXXXXXXXXXXXXXXX\""
 *     + "}";
 *
 * TransactionLinkResponse response = AbstractResponse.fromJson(jsonResponse, TransactionLinkResponse.class);
 * System.out.println("Status: " + response.getStatus());
 * System.out.println("Status Code: " + response.getStatusCode());
 * System.out.println("Merchant Reference ID: " + response.getMerchantReferenceId());
 * System.out.println("Transaction URL: " + response.getUrl());
 * </pre>
 */
@Data
@EqualsAndHashCode(callSuper = true)
public class TransactionLinkResponse extends AbstractResponse {
    /**
     * The status of the LINK transaction (e.g., SUCCESS, PENDING, FAILED).
     */
    private String status;

    /**
     * The status code returned by the LINK transaction API.
     * <p>Indicates the result of the operation (e.g., 200 for success, 400 for errors).</p>
     */
    private String statusCode;

    /**
     * The merchant's reference ID for the LINK transaction.
     */
    private String merchantReferenceId;

    /**
     * The URL provided for the LINK transaction.
     * <p>This URL typically redirects to a page for completing the transaction.</p>
     */
    private String url;

    /**
     * Default constructor
     */
    public TransactionLinkResponse() {}
}