package pro.mypvit.responses;

import lombok.Data;
import lombok.EqualsAndHashCode;

/**
 * Represents the response for a REST transaction request in the MyPVit SDK.
 *
 * <p>This class extends {@link AbstractResponse} to inherit functionality for JSON deserialization
 * and provides specific fields related to REST transaction results.</p>
 *
 * Key Features:
 * <ul>
 *     <li>Inherits JSON deserialization capabilities from {@link AbstractResponse}.</li>
 *     <li>Includes fields for transaction status, operator, references, and additional details.</li>
 *     <li>Supports automatic generation of getters, setters, equals, and hashCode methods through Lombok.</li>
 * </ul>
 *
 * Usage:
 * Use the `fromJson` method provided by {@link AbstractResponse} to deserialize a JSON response into
 * an instance of this class.
 *
 * Example:
 * <pre>
 * String jsonResponse = "{"
 *     + "\"status\": \"SUCCESS\","
 *     + "\"status_code\": \"200\","
 *     + "\"operator\": \"AIRTEL_MONEY\","
 *     + "\"reference_id\": \"REF123456\","
 *     + "\"merchant_reference_id\": \"MERCHANT_REF123\","
 *     + "\"merchant_operation_account_code\": \"ACC123456\","
 *     + "\"message\": \"Transaction processed successfully\""
 *     + "}";
 *
 * TransactionRestResponse response = AbstractResponse.fromJson(jsonResponse, TransactionRestResponse.class);
 * System.out.println("Status: " + response.getStatus());
 * System.out.println("Operator: " + response.getOperator());
 * System.out.println("Reference ID: " + response.getReferenceId());
 * System.out.println("Message: " + response.getMessage());
 * </pre>
 */
@Data
@EqualsAndHashCode(callSuper = true)
public class TransactionRestResponse extends AbstractResponse {
    /**
     * The status of the REST transaction (e.g., SUCCESS, PENDING, FAILED).
     */
    private String status;

    /**
     * The status code returned by the REST transaction API.
     * <p>Indicates the result of the operation (e.g., 200 for success, 400 for errors).</p>
     */
    private String statusCode;

    /**
     * The operator involved in the transaction (e.g., AIRTEL_MONEY, MOOV_MONEY).
     */
    private String operator;

    /**
     * The reference ID assigned to the transaction by the system.
     */
    private String referenceId;

    /**
     * The merchant's reference ID for the transaction.
     */
    private String merchantReferenceId;

    /**
     * The merchant's operation account code used in the transaction.
     */
    private String merchantOperationAccountCode;

    /**
     * The message returned by the REST transaction API.
     * <p>Provides additional information about the transaction result (e.g., success or error details).</p>
     */
    private String message;

    /**
     * Default constructor
     */
    public TransactionRestResponse() {}
}
