package pro.respawn.kmmutils.inputforms

/**
 * An error that occurred during validation.
 * Presence of ValidationErrors in the resulting [Input.Invalid] indicates
 */
public sealed interface ValidationError {

    /**
     * A value that is being validated
     */
    public val value: String

    /**
     * A generic validation error that can be subclassed. In this case,
     * you need to keep track of your custom errors yourself.
     */
    public abstract class Generic(
        override val value: String,
    ) : ValidationError

    /**
     * The input is blank.
     * @see pro.respawn.kmmutils.inputforms.default.Rules.NonEmpty
     */
    public data class Empty(
        override val value: String
    ) : ValidationError

    /**
     * Length of the input was not in [range]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.LengthInRange
     */
    public data class NotInRange(
        override val value: String,
        public val range: IntRange,
    ) : ValidationError

    /**
     * Input was shorter than [minLength]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.LongerThan
     */
    public data class TooShort(
        override val value: String,
        public val minLength: Int,
    ) : ValidationError

    /**
     * Input was longer than [maxLength]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.ShorterThan
     */
    public data class TooLong(
        override val value: String,
        public val maxLength: Int,
    ) : ValidationError

    /**
     * Input does not match the [pattern]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.Matches
     */
    public data class DoesNotMatch(
        override val value: String,
        val pattern: Regex,
    ) : ValidationError

    /**
     * Input does not start with [prefix]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.StartsWith
     */
    public data class DoesNotStartWith(
        override val value: String,
        public val prefix: String
    ) : ValidationError

    /**
     * Input does not end with [suffix]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.EndsWith
     */
    public data class DoesNotEndWith(
        override val value: String,
        public val suffix: String,
    ) : ValidationError

    /**
     * Input does not contain a [needle]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.Contains
     */
    public data class DoesNotContain(override val value: String, public val needle: String) : ValidationError

    /**
     * Input is not alpha-numeric
     * @see pro.respawn.kmmutils.inputforms.default.Rules.AlphaNumeric
     */
    public data class NotAlphaNumeric(override val value: String) : ValidationError

    /**
     * Input has digits
     * @see pro.respawn.kmmutils.inputforms.default.Rules.NoDigits
     */
    public data class ContainsDigits(override val value: String) : ValidationError

    /**
     * Input has no digits
     * @see pro.respawn.kmmutils.inputforms.default.Rules.HasDigit
     */
    public data class HasNoDigits(override val value: String) : ValidationError

    /**
     * Input has letters
     * @see pro.respawn.kmmutils.inputforms.default.Rules.NoLetters
     */
    public data class ContainsLetters(override val value: String) : ValidationError

    /**
     * Input has no letters
     * @see pro.respawn.kmmutils.inputforms.default.Rules.HasLetter
     */
    public data class HasNoLetters(override val value: String) : ValidationError

    /**
     * Input's value was not equal to [other]
     * @see pro.respawn.kmmutils.inputforms.default.Rules.Equals
     */
    public data class IsNotEqual(override val value: String, val other: String) : ValidationError

    public companion object
}
