/*
 * Decompiled with CFR 0.152.
 */
package pro.taskana.common.api;

import java.time.DayOfWeek;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.LongStream;
import pro.taskana.common.api.CustomHoliday;
import pro.taskana.common.api.exceptions.SystemException;

public class WorkingDaysToDaysConverter {
    private static final long OFFSET_GOOD_FRIDAY = -2L;
    private static final long OFFSET_EASTER_MONDAY = 1L;
    private static final long OFFSET_ASCENSION_DAY = 39L;
    private static final long OFFSET_WHIT_MONDAY = 50L;
    private static final long OFFSET_CORPUS_CHRISTI = 60L;
    private static final Set<CustomHoliday> GERMAN_HOLIDAYS = new HashSet<CustomHoliday>(Arrays.asList(CustomHoliday.of(1, 1), CustomHoliday.of(1, 5), CustomHoliday.of(3, 10), CustomHoliday.of(25, 12), CustomHoliday.of(26, 12)));
    private final boolean germanHolidaysEnabled;
    private final boolean corpusChristiEnabled;
    private final Set<CustomHoliday> customHolidays;
    private final EasterCalculator easterCalculator;

    public WorkingDaysToDaysConverter(boolean germanHolidaysEnabled, boolean corpusChristiEnabled) {
        this(germanHolidaysEnabled, corpusChristiEnabled, Collections.emptySet());
    }

    public WorkingDaysToDaysConverter(boolean germanHolidaysEnabled, boolean corpusChristiEnabled, Collection<CustomHoliday> customHolidays) {
        this.germanHolidaysEnabled = germanHolidaysEnabled;
        this.corpusChristiEnabled = corpusChristiEnabled;
        this.customHolidays = new HashSet<CustomHoliday>(customHolidays);
        this.easterCalculator = new EasterCalculator();
    }

    public Instant addWorkingDaysToInstant(Instant instant, Duration workingDays) {
        long days = this.convertWorkingDaysToDays(instant, workingDays.toDays(), ZeroDirection.ADD_DAYS);
        return instant.plus(Duration.ofDays(days));
    }

    public Instant subtractWorkingDaysFromInstant(Instant instant, Duration workingDays) {
        long days = this.convertWorkingDaysToDays(instant, -workingDays.toDays(), ZeroDirection.SUB_DAYS);
        return instant.plus(Duration.ofDays(days));
    }

    public boolean hasWorkingDaysInBetween(Instant left, Instant right) {
        long days = Duration.between(left, right).abs().toDays();
        Instant firstInstant = left.isBefore(right) ? left : right;
        return LongStream.range(1L, days).anyMatch(day -> this.isWorkingDay(firstInstant.plus(day, ChronoUnit.DAYS)));
    }

    public boolean isWorkingDay(Instant referenceDate) {
        LocalDate dateToCheck = LocalDateTime.ofInstant(referenceDate, ZoneId.systemDefault()).toLocalDate();
        return !this.isWeekend(dateToCheck) && !this.isHoliday(dateToCheck);
    }

    public boolean isWeekend(LocalDate dateToCheck) {
        return dateToCheck.getDayOfWeek().equals(DayOfWeek.SATURDAY) || dateToCheck.getDayOfWeek().equals(DayOfWeek.SUNDAY);
    }

    public boolean isHoliday(LocalDate date) {
        if (this.germanHolidaysEnabled && this.isGermanHoliday(date)) {
            return true;
        }
        return this.customHolidays.contains(CustomHoliday.of(date.getDayOfMonth(), date.getMonthValue()));
    }

    public boolean isGermanHoliday(LocalDate date) {
        if (GERMAN_HOLIDAYS.contains(CustomHoliday.of(date.getDayOfMonth(), date.getMonthValue()))) {
            return true;
        }
        long diffFromEasterSunday = ChronoUnit.DAYS.between(this.easterCalculator.getEasterSunday(date.getYear()), date);
        LongStream.Builder builder = LongStream.builder().add(-2L).add(1L).add(39L).add(50L);
        if (this.corpusChristiEnabled) {
            builder.add(60L);
        }
        return builder.build().anyMatch(c -> c == diffFromEasterSunday);
    }

    private long convertWorkingDaysToDays(Instant startTime, long numberOfDays, ZeroDirection zeroDirection) {
        if (startTime == null) {
            throw new SystemException("Internal Error: convertWorkingDaysToDays was called with a null startTime");
        }
        int direction = this.calculateDirection(numberOfDays, zeroDirection);
        long limit = Math.abs(numberOfDays);
        return LongStream.iterate(0L, i -> i + (long)direction).filter(day -> this.isWorkingDay(startTime.plus(day, ChronoUnit.DAYS))).skip(limit).findFirst().orElse(0L);
    }

    private int calculateDirection(long numberOfDays, ZeroDirection zeroDirection) {
        if (numberOfDays == 0L) {
            return zeroDirection.getDirection();
        }
        return numberOfDays >= 0L ? 1 : -1;
    }

    public String toString() {
        return "WorkingDaysToDaysConverter [germanHolidaysEnabled=" + this.germanHolidaysEnabled + ", corpusChristiEnabled=" + this.corpusChristiEnabled + ", customHolidays=" + this.customHolidays + ", easterCalculator=" + this.easterCalculator + "]";
    }

    static class EasterCalculator {
        LocalDate cachedEasterDay;

        EasterCalculator() {
        }

        LocalDate getEasterSunday(int year) {
            if (this.cachedEasterDay != null && this.cachedEasterDay.getYear() == year) {
                return this.cachedEasterDay;
            }
            int a = year % 19;
            int b = year / 100;
            int c = year % 100;
            int d = b / 4;
            int e = b % 4;
            int f = (b + 8) / 25;
            int g = (b - f + 1) / 3;
            int h = (19 * a + b - d - g + 15) % 30;
            int i = c / 4;
            int k = c % 4;
            int l = (32 + 2 * e + 2 * i - h - k) % 7;
            int m = (a + 11 * h + 22 * l) / 451;
            int n = h + l - 7 * m + 114;
            int month = n / 31;
            int day = n % 31 + 1;
            this.cachedEasterDay = LocalDate.of(year, month, day);
            return this.cachedEasterDay;
        }
    }

    private static enum ZeroDirection {
        SUB_DAYS(-1),
        ADD_DAYS(1);

        private final int direction;

        private ZeroDirection(int direction) {
            this.direction = direction;
        }

        public int getDirection() {
            return this.direction;
        }
    }
}

