/*
 * Decompiled with CFR 0.152.
 */
package pro.taskana.common.internal.util;

import java.time.DayOfWeek;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.LongStream;
import pro.taskana.common.api.CustomHoliday;
import pro.taskana.common.api.exceptions.InvalidArgumentException;
import pro.taskana.common.api.exceptions.SystemException;

public final class WorkingDaysToDaysConverter {
    private static final long OFFSET_GOOD_FRIDAY = -2L;
    private static final long OFFSET_EASTER_MONDAY = 1L;
    private static final long OFFSET_ASCENSION_DAY = 39L;
    private static final long OFFSET_WHIT_MONDAY = 50L;
    private static final long OFFSET_CORPUS_CHRISTI = 60L;
    private static final Set<CustomHoliday> GERMAN_HOLIDAYS = new HashSet<CustomHoliday>(Arrays.asList(CustomHoliday.of(1, 1), CustomHoliday.of(1, 5), CustomHoliday.of(3, 10), CustomHoliday.of(25, 12), CustomHoliday.of(26, 12)));
    private static boolean germanHolidaysEnabled;
    private static boolean corpusChristiEnabled;
    private static Set<CustomHoliday> customHolidays;
    private Instant referenceDate;
    private LocalDate easterSunday;

    private WorkingDaysToDaysConverter(Instant referenceDate) {
        this.easterSunday = WorkingDaysToDaysConverter.getEasterSunday(LocalDateTime.ofInstant(referenceDate, ZoneId.systemDefault()).getYear());
        this.referenceDate = referenceDate;
    }

    public Instant getReferenceDate() {
        return this.referenceDate;
    }

    public static WorkingDaysToDaysConverter initialize() {
        try {
            return WorkingDaysToDaysConverter.initialize(Instant.now());
        }
        catch (InvalidArgumentException ex) {
            throw new SystemException("Internal error. Cannot initialize WorkingDaysToDaysConverter. This should not happen", ex);
        }
    }

    public static WorkingDaysToDaysConverter initialize(Instant referenceDate) throws InvalidArgumentException {
        if (referenceDate == null) {
            throw new InvalidArgumentException("ReferenceDate cannot be used as NULL-Parameter");
        }
        return new WorkingDaysToDaysConverter(referenceDate);
    }

    public static void setGermanPublicHolidaysEnabled(boolean germanPublicHolidaysEnabled) {
        germanHolidaysEnabled = germanPublicHolidaysEnabled;
    }

    public static void setCorpusChristiEnabled(boolean corpusChristiEnabled) {
        WorkingDaysToDaysConverter.corpusChristiEnabled = corpusChristiEnabled;
    }

    public static void setCustomHolidays(List<CustomHoliday> holidays) {
        customHolidays = new HashSet(holidays == null ? Collections.emptyList() : holidays);
    }

    public long convertWorkingDaysToDays(Instant startTime, long numberOfDays) {
        return this.convertWorkingDaysToDays(startTime, numberOfDays, ZeroDirection.ADD_DAYS);
    }

    public long convertWorkingDaysToDays(Instant startTime, long numberOfDays, ZeroDirection zeroDirection) {
        if (startTime == null) {
            throw new SystemException("Internal Error: convertWorkingDaysToDays was called with a null startTime");
        }
        if (!startTime.equals(this.referenceDate)) {
            this.refreshReferenceDate(this.referenceDate);
        }
        int direction = this.calculateDirection(numberOfDays, zeroDirection);
        long limit = Math.abs(numberOfDays);
        return LongStream.iterate(0L, i -> i + (long)direction).filter(day -> this.isWorkingDay(day, startTime)).skip(limit).findFirst().orElse(0L);
    }

    public Instant addWorkingDaysToInstant(Instant instant, Duration workingDays) {
        long days = this.convertWorkingDaysToDays(instant, workingDays.toDays(), ZeroDirection.ADD_DAYS);
        return instant.plus(Duration.ofDays(days));
    }

    public Instant subtractWorkingDaysFromInstant(Instant instant, Duration workingDays) {
        long days = this.convertWorkingDaysToDays(instant, -workingDays.toDays(), ZeroDirection.SUB_DAYS);
        return instant.plus(Duration.ofDays(days));
    }

    public boolean hasWorkingDaysInBetween(Instant left, Instant right) {
        long days = Duration.between(left, right).abs().toDays();
        Instant firstInstant = left.isBefore(right) ? left : right;
        return LongStream.range(1L, days).anyMatch(day -> this.isWorkingDay(day, firstInstant));
    }

    public boolean isWorkingDay(long day, Instant referenceDate) {
        LocalDateTime dateToCheck = LocalDateTime.ofInstant(referenceDate, ZoneId.systemDefault()).plusDays(day);
        return !this.isWeekend(dateToCheck) && !this.isHoliday(dateToCheck.toLocalDate());
    }

    public boolean isWeekend(LocalDateTime dateToCheck) {
        return dateToCheck.getDayOfWeek().equals(DayOfWeek.SATURDAY) || dateToCheck.getDayOfWeek().equals(DayOfWeek.SUNDAY);
    }

    public boolean isHoliday(LocalDate date) {
        if (germanHolidaysEnabled && this.isGermanHoliday(date)) {
            return true;
        }
        return customHolidays.contains(CustomHoliday.of(date.getDayOfMonth(), date.getMonthValue()));
    }

    public boolean isGermanHoliday(LocalDate date) {
        if (GERMAN_HOLIDAYS.contains(CustomHoliday.of(date.getDayOfMonth(), date.getMonthValue()))) {
            return true;
        }
        long diffFromEasterSunday = ChronoUnit.DAYS.between(this.easterSunday, date);
        LongStream.Builder builder = LongStream.builder().add(-2L).add(1L).add(39L).add(50L);
        if (corpusChristiEnabled) {
            builder.add(60L);
        }
        return builder.build().anyMatch(c -> c == diffFromEasterSunday);
    }

    static LocalDate getEasterSunday(int year) {
        int a = year % 19;
        int b = year % 4;
        int c = year % 7;
        int k = year / 100;
        int p = (13 + 8 * k) / 25;
        int q = k / 4;
        int m = (15 - p + k - q) % 30;
        int n = (4 + k - q) % 7;
        int d = (19 * a + m) % 30;
        int e = (2 * b + 4 * c + 6 * d + n) % 7;
        if (d == 29 && e == 6) {
            return LocalDate.of(year, 3, 15).plusDays((long)d + (long)e);
        }
        if (d == 28 && e == 6 && (11 * m + 11) % 30 < 19) {
            return LocalDate.of(year, 3, 15).plusDays((long)d + (long)e);
        }
        return LocalDate.of(year, 3, 22).plusDays((long)d + (long)e);
    }

    void refreshReferenceDate(Instant newReferenceDate) {
        int yearOfNewReferenceDate;
        int yearOfReferenceDate = LocalDateTime.ofInstant(this.referenceDate, ZoneId.systemDefault()).getYear();
        if (yearOfReferenceDate != (yearOfNewReferenceDate = LocalDateTime.ofInstant(newReferenceDate, ZoneId.systemDefault()).getYear())) {
            this.easterSunday = WorkingDaysToDaysConverter.getEasterSunday(yearOfNewReferenceDate);
        }
        this.referenceDate = newReferenceDate;
    }

    private int calculateDirection(long numberOfDays, ZeroDirection zeroDirection) {
        if (numberOfDays == 0L) {
            return zeroDirection.getDirection();
        }
        return numberOfDays >= 0L ? 1 : -1;
    }

    public String toString() {
        return "WorkingDaysToDaysConverter [referenceDate=" + this.referenceDate + ", easterSunday=" + this.easterSunday + "]";
    }

    static {
        customHolidays = new HashSet<CustomHoliday>();
    }

    private static enum ZeroDirection {
        SUB_DAYS(-1),
        ADD_DAYS(1);

        private final int direction;

        private ZeroDirection(int direction) {
            this.direction = direction;
        }

        public int getDirection() {
            return this.direction;
        }
    }
}

