/*
 * Copyright 2021 Michelin CERT (https://cert.michelin.com/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.michelin.cert.redscan.utils.models.reports;

import com.michelin.cert.redscan.utils.datalake.DatalakeStorageException;
import com.michelin.cert.redscan.utils.datalake.DatalakeStorageItem;
import com.michelin.cert.redscan.utils.json.JsonUtils;
import com.michelin.cert.redscan.utils.models.Sendable;

import java.util.Arrays;

import kong.unirest.json.JSONObject;

import org.apache.commons.codec.digest.DigestUtils;

/**
 * Vulnerability model.
 *
 * @author Maxime ESCOURBIAC
 */
public class Vulnerability extends DatalakeStorageItem implements Sendable {

  private int severity;
  private String id;
  private String summary;
  private String description;
  private String url;
  private String origin;
  private String[] tags;

  /**
   * Vulnerability Constructor.
   */
  public Vulnerability() {
    this.index = "vulnerabilities";
    this.severity = Severity.INFO;
    this.id = "";
    this.summary = "";
    this.description = "";
    this.url = "";
    this.origin = "";
    this.tags = new String[0];
  }

  /**
   * Vulnerability Constructor.
   *
   * @param id Internal ID.
   * @param severity Vulnerability severity.
   * @param summary Vulnerability title.
   * @param description Detailed description of the vulnerability.
   * @param url Vulnerability URL.
   * @param origin Origin of the vulnerability (ex: redscan-urlcrazy).
   * @param tags Tags used by Mantisbt.
   */
  public Vulnerability(String id, int severity, String summary, String description, String url, String origin, String[] tags) {
    this.severity = severity;
    this.id = id;
    this.summary = summary;
    this.description = description;
    this.url = url;
    this.origin = origin;
    this.tags = tags;
  }

  /**
   * Convert the service to json string.
   *
   * @return The string representation of the json object.
   */
  @Override
  public String toJson() {
    JSONObject jsonObject = new JSONObject();
    JsonUtils.setSafeString(jsonObject, "id", id);
    JsonUtils.setSafeInt(jsonObject, "severity", severity);
    JsonUtils.setSafeString(jsonObject, "summary", summary);
    JsonUtils.setSafeString(jsonObject, "description", description);
    JsonUtils.setSafeString(jsonObject, "url", url);
    JsonUtils.setSafeString(jsonObject, "origin", origin);
    JsonUtils.setSafeStringArray(jsonObject, "tags", tags);
    return jsonObject.toString();
  }

  @Override
  public void fromJson(String json) {
    JSONObject jsonObject = new JSONObject(json);
    this.id = JsonUtils.getSafeString(jsonObject, "id");
    this.severity = JsonUtils.getSafeInt(jsonObject, "severity");
    this.summary = JsonUtils.getSafeString(jsonObject, "summary");
    this.description = JsonUtils.getSafeString(jsonObject, "description");
    this.url = JsonUtils.getSafeString(jsonObject, "url");
    this.origin = JsonUtils.getSafeString(jsonObject, "origin");
    this.tags = JsonUtils.getSafeStringArray(jsonObject, "tags");
  }

  @Override
  public String getFanoutExchangeName() {
    return "com.michelin.cert.fanout.vulnerabilities";
  }

  @Override
  public boolean upsert() throws DatalakeStorageException {
    boolean result = (find() != null);
    if (result) {
      result &= this.upsertField("id", id);
      result &= this.upsertField("severity", severity);
      result &= this.upsertField("summary", summary);
      result &= this.upsertField("description", description);
      result &= this.upsertField("url", url);
      result &= this.upsertField("origin", origin);
      result &= this.upsertField("tags", tags);
    }
    return result;
  }

  @Override
  protected <T extends DatalakeStorageItem> T fromDatalake(JSONObject object) {
    return (object == null) ? null
        : (T) new Vulnerability(
            JsonUtils.getSafeString(object, "id"),
            JsonUtils.getSafeInt(object, "severity"),
            JsonUtils.getSafeString(object, "summary"),
            JsonUtils.getSafeString(object, "description"),
            JsonUtils.getSafeString(object, "url"),
            JsonUtils.getSafeString(object, "origin"),
            JsonUtils.getSafeStringArray(object, "tags"));
  }

  /**
   * *
   * Generate unique internal id from dedicated value. It should include the origin to avoid collision.
   *
   * @param values Values to differentiate the vulnerability.
   * @return Unique vulnerability id.
   */
  public static String generateId(String... values) {
    String result = "";
    if (values != null && values.length > 0) {
      result = DigestUtils.sha1Hex(Arrays.toString(values));
    }
    return result;
  }

  /**
   * Vulnerability severity.
   *
   * @return Vulnerability severity.
   */
  public int getSeverity() {
    return severity;
  }

  /**
   * Vulnerability severity.
   *
   * @param severity Vulnerability severity.
   */
  public void setSeverity(int severity) {
    this.severity = severity;
  }

  /**
   * Internal ID.
   *
   * @return Internal ID.
   */
  public String getId() {
    return id;
  }

  /**
   * Internal ID.
   *
   * @param id Internal ID.
   */
  public void setId(String id) {
    this.id = id;
  }

  /**
   * Vulnerability title.
   *
   * @return Vulnerability title.
   */
  public String getSummary() {
    return summary;
  }

  /**
   * Vulnerability title.
   *
   * @param summary Vulnerability title.
   */
  public void setSummary(String summary) {
    this.summary = summary;
  }

  /**
   * Detailed description of the vulnerability.
   *
   * @return Detailed description of the vulnerability.
   */
  public String getDescription() {
    return description;
  }

  /**
   * Detailed description of the vulnerability.
   *
   * @param description Detailed description of the vulnerability.
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * Vulnerability URL.
   *
   * @return Vulnerability URL.
   */
  public String getUrl() {
    return url;
  }

  /**
   * Vulnerability URL.
   *
   * @param url Vulnerability URL.
   */
  public void setUrl(String url) {
    this.url = url;
  }

  /**
   * Origin of the vulnerability (ex: redscan-urlcrazy).
   *
   * @return Origin of the vulnerability (ex: redscan-urlcrazy).
   */
  public String getOrigin() {
    return origin;
  }

  /**
   * Origin of the vulnerability (ex: redscan-urlcrazy).
   *
   * @param origin Origin of the vulnerability (ex: redscan-urlcrazy).
   */
  public void setOrigin(String origin) {
    this.origin = origin;
  }

  /**
   * Tags used by Mantisbt.
   *
   * @return Tags used by Mantisbt.
   */
  public String[] getTags() {
    return tags;
  }

  /**
   * Tags used by Mantisbt.
   *
   * @param tags Tags used by Mantisbt.
   */
  public void setTags(String[] tags) {
    this.tags = tags;
  }
}
