package red.mulan.boot.common.result;

import lombok.Getter;
import red.mulan.boot.common.code.CommonCode;
import red.mulan.boot.common.utils.TraceIdUtils;

/**
 * 响应结果类<br>
 *
 * @author Max
 * @version 1.0.0
 * @since 2023/2/17 20:50
 */
@Getter
public class ResponseResult <T>  implements Result{

    /**
     * 返回的结构
     */
    private final T data;
    /**
     * 请求ID
     */
    private final String traceId;



    /**
     * 返回操作成功响应信息
     * @return 操作成功的响应信息
     * @param <R> 返回数据的泛型
     */
    public static <R> ResponseResult<R> ok(){
        return new ResponseResult<>(CommonCode.SUCCESS);
    }

    /**
     * 返回操作成功响应信息 并带有返回数据
     * @param data  返回的数据
     * @return 操作成功的响应信息
     * @param <R> 返回数据的泛型
     */
    public static <R> ResponseResult<R> ok(R data){
        return new ResponseResult<>(CommonCode.SUCCESS,data);
    }


    /**
     * 返回默认操作失败响应信息
     * @return 响应信息
     * @param <R> 响应信息泛型
     */
    public static <R> ResponseResult<R> fail(){
        return new ResponseResult<>(CommonCode.FAIL);
    }

    /**
     * 返回自定义操作失败响应信息 带数据返回
     * @param result 响应信息
     * @param data 返回的数据
     * @return 响应信息
     * @param <R> 响应信息泛型
     */
    public static <R> ResponseResult<R> fail(Result result,R data){
        return result.success() ? new ResponseResult<>(CommonCode.FAIL,data) : new ResponseResult<>(result,data);
    }

    /**
     * 返回自定义操作失败响应信息
     * @param result 返回的数据
     * @return 响应信息
     * @param <R> 响应信息泛型
     */
    public static <R> ResponseResult<R> fail(Result result){
        return result.success() ? new ResponseResult<>(CommonCode.FAIL) : new ResponseResult<>(result);
    }

    /**
     * 未授权
     * @return 响应未授权
     * @param <R> 返回数据泛型
     */
    public static <R> ResponseResult<R> unauthorized(){
        return new ResponseResult<>(CommonCode.UNAUTHORIZED);
    }

    /**
     * 返回权限不足
     * @return 响应权限不足
     * @param <R> 返回数据泛型
     */
    public static <R> ResponseResult<R> forbidden(){
        return new ResponseResult<>(CommonCode.FORBIDDEN);
    }




    public ResponseResult(Result result,T data) {
        this.code = result.code();
        this.success = result.success();
        this.message = result.message();
        this.data =  data;
        this.traceId = TraceIdUtils.get();
    }


    public ResponseResult(Result result) {
        this.code = result.code();
        this.success = result.success();
        this.message = result.message();
        this.data = null;
        this.traceId = TraceIdUtils.get();
    }






    /**
     * 自定义状态码
     */
    private final String code;

    /**
     * 是否成功
     */
    private final boolean success;

    /**
     * 响应信息
     */
    private final String message;


    /**
     * 返回自定义请求状态码
     *
     * @return 返回自定义状态码
     */
    @Override
    public String code() {
        return this.code;
    }

    /**
     * 返回是否成功
     *
     * @return 是否成功
     */
    @Override
    public Boolean success() {
        return this.success;
    }

    /**
     * 返回提示信息
     *
     * @return 提示信息
     */
    @Override
    public String message() {
        return this.message;
    }
}
