package red.mulan.boot.common.handler;

import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * bean 处理<br>
 *
 * @author Max
 * @version 1.0.0
 * @since 2023/2/23 20:41
 */
@Slf4j
public class SpringBeanHandler{

    /**
     * spring 上下文
     */
    @Resource
    private ApplicationContext applicationContext;


    /**
     * 可重入读写锁
     */
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();


    /**
     * 注册bean
     * @param name bean 名称
     * @param clazz 类型
     * @param args bean的参数
     */
    public void registerBean( String name, Class<?> clazz, Map<String,Object> args) {
        if(applicationContext.containsBean(name)) {
            return;
        }
        ConfigurableApplicationContext configurableApplicationContext = (ConfigurableApplicationContext)applicationContext;
        BeanDefinitionBuilder beanDefinitionBuilder = BeanDefinitionBuilder.genericBeanDefinition(clazz);
        if (!args.isEmpty()){
            Set<Map.Entry<String, Object>> entries = args.entrySet();
            for (Map.Entry<String, Object> next : entries) {
                beanDefinitionBuilder.addPropertyValue(next.getKey(), next.getValue());
            }
        }
        BeanDefinition beanDefinition = beanDefinitionBuilder.getRawBeanDefinition();
        BeanDefinitionRegistry beanFactory = (BeanDefinitionRegistry) configurableApplicationContext.getBeanFactory();
        beanFactory.registerBeanDefinition(name, beanDefinition);
    }



    public Object getBean(String beanName){
        if (StringUtils.hasText(beanName)){
            return applicationContext.getBean(beanName);
        }
        return null;

    }


    /**
     * 根据bean名称来删除bean
     * @param beanName bean名称
     */
    public void removeBean(String beanName) {
        lock.writeLock().lock();
        try {

            DefaultListableBeanFactory beanFactory = getBeanFactory();
            if (beanFactory.containsBeanDefinition(beanName)) {
                beanFactory.destroySingleton(beanName);
                beanFactory.removeBeanDefinition(beanName);
                log.info("注销[{}]成功！", beanName);
            } else {
                log.info("不存在[{}]，不需要注销！", beanName);
            }
        } finally {
            lock.writeLock().unlock();
        }
    }


    /**
     * bean工厂
     * @return bean工厂
     */
    private DefaultListableBeanFactory getBeanFactory() {
        ConfigurableApplicationContext configurableApplicationContext = (ConfigurableApplicationContext) applicationContext;
        return (DefaultListableBeanFactory) configurableApplicationContext.getBeanFactory();
    }
}
