/*
 * Click nbfs://nbhost/SystemFileSystem/Templates/Licenses/license-default.txt to change this license
 * Click nbfs://nbhost/SystemFileSystem/Templates/Classes/Class.java to edit this template
 */
package rocks.imsofa.ai.puppychatter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.slf4j.LoggerFactory;
import rocks.imsofa.ai.puppychatter.cache.CacheService;

/**
 *
 * @author USER
 */
public abstract class AbstractPuppyChatter<T extends PromptParameters, S extends Response> implements PuppyChatter<T, S> {

    protected Map<String, List<Conversation>> sessionHistory = new HashMap<>();
    protected CacheService cacheService = null;
    protected String replyRole = "assistant";

    public AbstractPuppyChatter(CacheService cacheService, String replyRole) {
        this.cacheService = cacheService;
        this.replyRole = replyRole;
    }

    public AbstractPuppyChatter() {
    }

    @Override
    public String createSession() {
        String sessionId = UUID.randomUUID().toString();
        sessionHistory.put(sessionId, new ArrayList<>());
        return sessionId;
    }

    @Override
    public S bark(String sessionId, String prompt, T parameters) throws BarkException {
        return this.bark(sessionId, prompt, parameters, (r) -> {
            return (!r.isError()) ? VerificationResult.GOOD : VerificationResult.GIVE_UP;
        });
    }

    public void bark(String sessionId, String prompt, T parameters, BarkCallback barkCallback) throws BarkException {
        this._bark(sessionId, prompt, parameters, (r) -> {
            return (!r.isError()) ? VerificationResult.GOOD : VerificationResult.GIVE_UP;
        }, barkCallback);
    }

    @Override
    public S bark(String sessionId, String prompt, T parameters, ResponseVerifier verifier) throws BarkException {
        return this._bark(sessionId, prompt, parameters, verifier, null);
    }

    @Override
    public void closeSession(String sessionId) {
        sessionHistory.remove(sessionId);
    }

    /**
     * merge everything to make it simpler
     *
     * @param sessionId
     * @param prompt
     * @param parameters
     * @param verifier
     * @param barkCallback
     * @return
     * @throws Exception
     */
    private S _bark(String sessionId, String prompt, T parameters, ResponseVerifier verifier, BarkCallback barkCallback) throws BarkException {
        List<Conversation> messages = sessionHistory.get(sessionId);
        Conversation newMessage = new Conversation(parameters.getRole(), prompt);

        messages.add(newMessage);
        //visit cache in advance
        if (cacheService != null) {
            try {
                messages = cacheService.sync2BoundConversations(messages);
            } catch (Exception ex) {
                throw new BarkException(ex);
            }
        }
        sessionHistory.put(sessionId, messages);
        if (cacheService != null) {
            try {
                Conversation c = cacheService.getCachedReply(messages);
                if (c != null) {
                    S response = createResponseFromConversation(c);
                    if (verifier.verify(response).equals(VerificationResult.GOOD)) {
                        LoggerFactory.getLogger(getClass()).info("reply obtained from cache");
                        messages.add(c);
                        if (barkCallback != null) {
                            barkCallback.responseChunkReceived(response);
                            return null;
                        }else{
                            return response;
                        }
                    }
                }
            } catch (Exception ex) {
                throw new BarkException(ex);
            }
        }
        LoggerFactory.getLogger(getClass()).info("cache not available, bark at the underlying service");
        if (barkCallback != null) {
            try {
                //then use async way
                this._bark(sessionId, messages, parameters, barkCallback);
            } catch (Exception ex) {
                Logger.getLogger(AbstractPuppyChatter.class.getName()).log(Level.SEVERE, null, ex);
            }
            return null;
        } else {
            //then use sync way
            S response = null;
            try {
                response = this._bark(sessionId, messages, parameters);
            } catch (Exception ex) {
                throw new BarkException(ex);
            }
//            System.out.println(response.getMessage());
            VerificationResult responseQuality = verifier.verify(response);
            LoggerFactory.getLogger(getClass()).info("verification result: " + responseQuality);
            while (responseQuality.equals(VerificationResult.TRY_AGAIN)) {
                try {
                    response = this._bark(sessionId, messages, parameters);
                    responseQuality = verifier.verify(response);
                    LoggerFactory.getLogger(getClass()).info("verification result: " + responseQuality);
                } catch (Exception ex) {
                    throw new BarkException(ex);
                }
            }
            if (responseQuality.equals(VerificationResult.GOOD)) {
                Conversation replyMessage = new Conversation(replyRole, response.getMessage());
//                LoggerFactory.getLogger(getClass()).info("saving to cache");
                //save to cache
                if (cacheService != null) {
//                    LoggerFactory.getLogger(getClass()).info("saving to cache 2");
                    try {
                        messages = cacheService.sync2BoundConversations(messages);
                        cacheService.cacheReply(messages, replyMessage);
                        
                    } catch (Exception ex) {
                        throw new BarkException(ex);
                    }
                }
                messages.add(replyMessage);
                sessionHistory.put(sessionId, messages);
                return response;
            } else {
                response.setError(true);
                throw new BarkException(response);
            }
        }
    }

    protected abstract S _bark(String sessionId, List<Conversation> messages, T parameters) throws Exception;

    protected abstract void _bark(String sessionId, List<Conversation> messages, T parameters, BarkCallback callback) throws Exception;
    
    protected abstract T createDefaultPromptParameter();

    @Override
    public void bark(String sessionId, String prompt, BarkCallback barkCallback) throws BarkException {
        this.bark(sessionId, prompt, this.createDefaultPromptParameter(), barkCallback);
    }

    @Override
    public S bark(String sessionId, String prompt) throws BarkException {
        return this.bark(sessionId, prompt, this.createDefaultPromptParameter());
    }

    @Override
    public S bark(String sessionId, String prompt, ResponseVerifier verifier) throws BarkException {
        return this.bark(sessionId, prompt, this.createDefaultPromptParameter(), verifier);
    }
    
    

    protected abstract S createResponseFromConversation(Conversation conversation);
}
