package rocks.imsofa.ai.puppychatter.openai;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.jxpath.JXPathContext;
import org.slf4j.LoggerFactory;

import com.github.kevinsawicki.http.HttpRequest;
import com.google.gson.Gson;
import com.hankcs.hanlp.dependency.nnparser.util.Log;

import rocks.imsofa.ai.puppychatter.BarkCallback;
import rocks.imsofa.ai.puppychatter.Conversation;
import rocks.imsofa.ai.puppychatter.Response;
import rocks.imsofa.ai.puppychatter.cache.CacheService;

/**
 * an implementation of OpenAICompatiblePromptParameters that uses an
 * InputStream to process the response
 */
@SuppressWarnings("all")
public abstract class OpenAICompatibleInputStreamPuppyChatter<S extends OpenAICompatiblePromptParameters, T extends Response>
        extends OpenAICompatiblePuppyChatter<S, T> {
    public OpenAICompatibleInputStreamPuppyChatter() {
        super();
    }

    public OpenAICompatibleInputStreamPuppyChatter(String replyRole) {
        super(null, replyRole);
    }

    public OpenAICompatibleInputStreamPuppyChatter(CacheService cacheService, String replyRole) {
        super(cacheService, replyRole);
    }

    @Override
    protected T _bark(String sessionId, List<Conversation> messages, S parameters) throws Exception {
        StreamResultIterator it = getStreamResultIterator(sessionId, messages, parameters, false);
        return processResponseStreamFromLLMProviders(sessionId, it, messages, parameters, null, false);
    }

    @Override
    protected void _bark(String sessionId, List<Conversation> messages, S parameters, BarkCallback<T> callback)
            throws Exception {

        if (parameters.getJsonSchema() != null || (parameters.getTools() != null && parameters.getTools().size() > 0)) {
            // in these cases streaming is very difficult
            StreamResultIterator it = getStreamResultIterator(sessionId, messages, parameters, false);
            processResponseStreamFromLLMProviders(sessionId, it, messages, parameters, callback, false);
        } else {
            StreamResultIterator it = getStreamResultIterator(sessionId, messages, parameters, true);
            processResponseStreamFromLLMProviders(sessionId, it, messages, parameters, callback, true);
        }

    }

    protected abstract StreamResultIterator getStreamResultIterator(String sessionId, List<Conversation> messages,
            S parameters, boolean streamed) throws Exception;

    protected T processResponseStreamFromLLMProviders(
            String sessionId,
            StreamResultIterator it,
            List<Conversation> messages,
            S parameters,
            BarkCallback<T> callback,
            boolean streamed) throws Exception {
        Response response = new Response();
        response.setLastPrompt(List.copyOf(messages));
        response.setMessage("");
        String ret = null;
        Gson gson = new Gson();
        StringBuilder sb = new StringBuilder();
        try {
            while (it.hasNext()) {
                String line = it.next();
                if (callback != null && streamed) {
                    // System.out.println("line="+line);
                    // streamed response
                    Response chunk = new Response();
                    Map result = gson.fromJson(line, Map.class);
                    JXPathContext context = JXPathContext.newContext(result);
                    String output = (String) context.getValue("choices[1]/delta/content");
                    chunk.setMessage(output);
                    response.setMessage(response.getMessage() + output);
                    callback.responseChunkReceived((T) chunk, false);
                }
                // non-streamed response and final verification
                sb.append(line).append("\r\n");

            }
            response.setMessage(sb.toString());
            if (callback != null && streamed) {
                // System.out.println("streamed");
                // streamed response
                Response chunk = new Response();
                chunk.setMessage(null);
                callback.responseChunkReceived((T) chunk, true);
                callback.finalVerificationResult(parameters.getResponseVerifier().verify(response));
            } else {
                // System.out.println("non streamed");
                // non-streamed response
                ret = sb.toString();
                // System.out.println("original returned message: " + ret);
                LoggerFactory.getLogger(this.getClass()).debug("original returned message: " + ret);
                Map result = gson.fromJson(ret, Map.class);
                JXPathContext context = JXPathContext.newContext(result);
                String output = (String) context.getValue("choices[1]/message/content");
                if (parameters.getTools() != null) {
                    Map<String, ToolCallRequest> toolCallRequestMap = this.toolCallRequests.get(sessionId);
                    if (toolCallRequestMap == null) {
                        toolCallRequestMap = new HashMap<>();
                        toolCallRequests.put(sessionId, toolCallRequestMap);
                    }
                    List<Map> toolCalls = (List) context.getValue("choices[1]/message/tool_calls");
                    if (toolCalls != null && toolCalls.size() > 0) {
                        // create a ToolCallRequestConversation
                        ToolCallRequestConversation toolCallConversation = new ToolCallRequestConversation();
                        toolCallConversation.setRole("assistant");
                        toolCallConversation.setContent("");

                        // tool calls
                        List<ToolCallRequest> toolCallRequests = new ArrayList<>();
                        toolCallConversation.setTool_calls(toolCallRequests);
                        ToolCallRequest processingToolCallRequest = null;
                        for (int i = 0; i < toolCalls.size(); i++) {
                            Map toolCall = toolCalls.get(i);
                            ToolCallRequest toolCallObj = null;
                            if (toolCallRequestMap.containsKey(toolCall.get("id")) == false) {
                                // the tool call has not already been registered
                                toolCallObj = new ToolCallRequest();
                                toolCallObj.setIndex(i);
                                toolCallObj.setId((String) toolCall.get("id"));
                                Map functionMap = (Map) toolCall.get("function");
                                ToolCallRequest.Function function = new ToolCallRequest.Function();
                                function.setName((String) functionMap.get("name"));
                                // function.setArguments((Map) gson.fromJson((String)
                                // functionMap.get("arguments"), Map.class));
                                function.setArguments((String) functionMap.get("arguments"));
                                toolCallObj.setFunction(function);
                            } else {
                                toolCallObj = toolCallRequestMap.get(toolCall.get("id"));
                            }
                            toolCallRequests.add(toolCallObj);
                            if (toolCallObj.getResponse() == null) {
                                // the tool call has not yet been processed
                                processingToolCallRequest = toolCallObj;
                                break;
                            }
                        }
                        // process tool calls

                        if (processingToolCallRequest != null) {
                            if (toolCallProcessor != null) {
                                String toolCallResponse = toolCallProcessor
                                        .processToolCallRequest(processingToolCallRequest);
                                processingToolCallRequest.setResponse(toolCallResponse);
                                ToolCallResponseConversation toolCallResponseConversation = new ToolCallResponseConversation(
                                        "tool",
                                        processingToolCallRequest.getResponse(),
                                        processingToolCallRequest.getId(),
                                        processingToolCallRequest.getFunction().getName());
                                messages.add(toolCallConversation);
                                messages.add(toolCallResponseConversation);
                                Response newResponse = this._bark(sessionId, messages, parameters);
                                // System.out.println(newResponse.getMessage());
                                output = newResponse.getMessage();
                            } else {
                                LoggerFactory.getLogger(getClass()).warn("tool call processor is not set");
                                throw new Exception("tool call exists but tool call processor is not set");
                            }
                        }

                    }
                }

                // String finishReason = (String) context.getValue("choices[1]/finish_reason");
                // Map message = (Map) context.getValue("choices[1]/message");

                response.setError(false);
                response.setErrorMessage(null);

                response.setMessage(output);
                if (callback != null && streamed == false) {
                    callback.responseChunkReceived((T) response, true);
                    callback.finalVerificationResult(parameters.getResponseVerifier().verify(response));
                }
                return (T) response;
            }
        } catch (Exception e) {
            LoggerFactory.getLogger(getClass()).error(e.getMessage() + "\r\n" + ret);
            response.setError(true);
            response.setErrorMessage(e.getMessage());
            return (T) response;
        }

        return null;
    }

    public static abstract class StreamResultIterator {
        protected InputStream inputStream;

        public StreamResultIterator(InputStream inputStream) {
            this.inputStream = inputStream;
        }

        public abstract String next();

        public abstract boolean hasNext();
    }
}
