package rocks.imsofa.ai.puppychatter.gemini;

import rocks.imsofa.ai.puppychatter.PuppyChatter;
import rocks.imsofa.ai.puppychatter.openai.OpenAICompatiblePuppyChatter;
import rocks.imsofa.ai.puppychatter.openrouter.OpenrouterPuppyChatter;

/**
 * Several functionalities in the gemini package require a PuppyChatter instance,
 * this class facilitates the initialization of that instance.
 * 
 * By default, this class will create an OpenrouterPuppyChatter instance with no cache service and use
 * the following environment variable to initialize it (via System.getenv (at first) or System.getProperty):
 * 
 * <ul>
 *  <li>PUPPYCHATTER_OPENROUTER_API_KEY</li>
 *  <li>PUPPYCHATTER_OPENROUTER_DEFAULT_MODEL</li>
 * </ul>
 * 
 * Users can also use the static method setPuppyChatter to set a custom OpenAICompatiblePuppyChatter instance.
 */
@SuppressWarnings("all")
public class InternalPuppyChatterInitializer {
    private static OpenAICompatiblePuppyChatter puppyChatter = null;

    public synchronized static void setPuppyChatter(OpenAICompatiblePuppyChatter puppyChatter) {
        InternalPuppyChatterInitializer.puppyChatter = puppyChatter;
    }

    public synchronized static OpenAICompatiblePuppyChatter getPuppyChatter() {
        if (puppyChatter == null) {
            String apiKey = System.getenv("PUPPYCHATTER_OPENROUTER_API_KEY");
            String defaultModel = System.getenv("PUPPYCHATTER_OPENROUTER_DEFAULT_MODEL");
            if(apiKey==null){
                apiKey = System.getProperty("PUPPYCHATTER_OPENROUTER_API_KEY");
            }
            if(defaultModel==null){
                defaultModel = System.getProperty("PUPPYCHATTER_OPENROUTER_DEFAULT_MODEL");
            }
            if (apiKey == null || defaultModel == null) {
                throw new IllegalStateException("PuppyChatter is not initialized. Please set the environment variables PUPPYCHATTER_OPENROUTER_API_KEY and PUPPYCHATTER_OPENROUTER_DEFAULT_MODEL.");
            }
            puppyChatter = new OpenrouterPuppyChatter(apiKey, defaultModel, null);
        }
        return puppyChatter;
    }
}
