package rocks.imsofa.sjl;

import com.github.kevinsawicki.http.HttpRequest;
import com.google.gson.Gson;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Logger;

public class Messenger {
    private static Messenger instance=null;
    private static boolean initialized=false;
    private ExecutorService executorService= Executors.newCachedThreadPool();
    private Map<ActionRequest, ActionResultCallback> pendingRequests=new HashMap<>();
    private Map<ActionCommand, ActionCallback> actionCommandToCallbackMap=new HashMap<>();
    private SimpleServer simpleServer=null;

    public synchronized void init(List<ActionCommand> providedActions) throws Exception {
        init(providedActions, 0);
    }
    /**
     * init the messenger
     * create a in-app server and register the port to
     * the server
     */
    public synchronized void init(List<ActionCommand> providedActions, int serverPort) throws Exception {
        //initialize the server and register with local sjlmessagebus
        if(initialized){
            return;
        }
        simpleServer=new SimpleServer(serverPort, (str, ctx) -> {
            Gson gson=new Gson();
            RemotePacket packet=gson.fromJson(str, RemotePacket.class);
            if(packet.getType()==RemotePacket.ACTION_RESPONSE){
                //if the RemotePacket contains an ActionResponse
                ActionResponse response=packet.getActionResponse();
                ActionRequest request=response.getSourceRequest();
                ActionResultCallback callback=pendingRequests.get(request);
                System.out.println("callback="+callback);
                if(callback!=null){
                    callback.onResult(request, response);
                    pendingRequests.remove(request);
                }
                ctx.close();
            }else if(packet.getType()==RemotePacket.ACTION_REQUEST){
                //if the RemotePacket contains an ActionRequest
                ActionRequest request=packet.getActionRequest();
                Logger.getLogger(this.getClass().getName()).info("action request received for "+request.getActionCommand());
                ActionCallback callback=actionCommandToCallbackMap.get(request.getActionCommand());
                ActionResponse actionResponse=null;
                if(callback!=null){
                    String response=callback.onExecute(request);
                    actionResponse=new ActionResponse(request, response, false);

                }else{
                    actionResponse=new ActionResponse(request, "{'message':'not supported'}", true);
                }
                String json=gson.toJson(actionResponse);
                ctx.writeAndFlush(json);
                Logger.getLogger(this.getClass().getName()).info("response for action "+request.getActionCommand()+" is written "+json);
                ctx.close();
            }
        });
        int port = simpleServer.getPort();
        ActionCommandRegistrationRequest request = new ActionCommandRegistrationRequest();
        request.setPort(port);
        request.setActionCommands(providedActions);
        Gson gson = new Gson();
        String json = gson.toJson(request);
        boolean ok = HttpRequest.post("http://localhost:8221/registerActionCommands")
                .contentType("application/json")
                .send(json)
                .ok();
        if (ok) {
            initialized = true;
        } else {
           simpleServer.shutdown();
           throw new Exception("unable to register action commands");
        }

    }

    /**
     * send a action command to the sjl bus
     * @param request
     * @throws Exception
     */
    public void execute(ActionRequest request, ActionResultCallback callback) throws Exception {
        if(!initialized){
            throw new Exception("messenger not initialized");
        }
        request.setSourcePort(simpleServer.getPort());
        executorService.submit(()->{
            Gson gson=new Gson();
            pendingRequests.put(request, callback);
            String json=gson.toJson(request);
            boolean ok=HttpRequest.post("http://localhost:8221/executeActionCommand")
                    .contentType("application/json")
                    .send(json)
                    .ok();
            if(!ok){
                pendingRequests.remove(request);
                callback.onError(request);
            }
        });
    }

    public void registerActionCallback(ActionCommand actionCommand, ActionCallback callback){
        actionCommandToCallbackMap.put(actionCommand, callback);
    }

    public static Messenger getInstance() {
        synchronized (Messenger.class) {
            if (instance == null) {
                instance = new Messenger();
            }
            return instance;
        }
    }

    public int getPort() {
        return simpleServer.getPort();
    }

    public void shutdown() {
        executorService.shutdown();
        try {
            simpleServer.shutdown();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) throws Exception {
        Messenger.getInstance().init(List.of(
                new ActionCommand("test", "test1")
        ), 8001);
        Messenger.getInstance().registerActionCallback(new ActionCommand("test", "test1"),
                new ActionCallback() {
                    @Override
                    public String onExecute(ActionRequest request) {
                        System.out.println("action executed");
                        return "hello";
                    }
                });
    }
}