/*
 * Decompiled with CFR 0.152.
 */
package rocks.inspectit.opencensus.influx;

import io.opencensus.common.Timestamp;
import io.opencensus.metrics.LabelKey;
import io.opencensus.metrics.LabelValue;
import io.opencensus.metrics.Metrics;
import io.opencensus.metrics.export.Distribution;
import io.opencensus.metrics.export.Metric;
import io.opencensus.metrics.export.MetricDescriptor;
import io.opencensus.metrics.export.Point;
import io.opencensus.stats.Stats;
import io.opencensus.stats.View;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.BatchPoints;
import org.influxdb.dto.Point;
import org.influxdb.dto.Query;
import org.influxdb.dto.QueryResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InfluxExporter
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(InfluxExporter.class);
    private final String url;
    private final String user;
    private final String password;
    private final String database;
    private final String retention;
    private InfluxDB influx;
    private boolean createDatabase;

    public InfluxExporter(String url, String user, String password, String database, String retention, boolean createDatabase) {
        this.url = url;
        this.user = user;
        this.password = password;
        this.database = database;
        this.retention = retention;
        this.createDatabase = createDatabase;
    }

    public synchronized void export() {
        List<Metric> metrics = Metrics.getExportComponent().getMetricProducerManager().getAllMetricProducer().stream().flatMap(mp -> mp.getMetrics().stream()).collect(Collectors.toList());
        this.export(metrics);
    }

    synchronized void export(Collection<Metric> metrics) {
        Map<String, View> viewsMap = Stats.getViewManager().getAllExportedViews().stream().collect(Collectors.toMap(v -> v.getName().asString(), v -> v));
        this.connectAndCreateDatabase();
        if (this.influx != null) {
            try {
                BatchPoints batch = BatchPoints.database((String)this.database).retentionPolicy(this.retention).build();
                metrics.stream().flatMap(m -> {
                    String metricName = m.getMetricDescriptor().getName();
                    View view = (View)viewsMap.get(metricName);
                    return this.toInfluxPoints((Metric)m, this.getMeasurementName(metricName, view), this.getFieldName(m.getMetricDescriptor().getType(), view));
                }).forEach(arg_0 -> ((BatchPoints)batch).point(arg_0));
                this.influx.write(batch);
            }
            catch (Exception e) {
                log.error("Error writing to InfluxDB", (Throwable)e);
            }
        }
    }

    String getMeasurementName(String metricName, View view) {
        if (view == null) {
            return this.sanitizeName(metricName);
        }
        String measureName = view.getMeasure().getName();
        return this.sanitizeName(measureName);
    }

    String getFieldName(MetricDescriptor.Type metricType, View view) {
        if (view == null) {
            return this.getDefaultFieldName(metricType);
        }
        String measureName = view.getMeasure().getName();
        String viewName = view.getName().asString();
        String fieldName = this.sanitizeName(this.removeCommonPrefix(viewName, measureName));
        if (fieldName.isEmpty()) {
            return this.getDefaultFieldName(metricType);
        }
        return fieldName;
    }

    private String getDefaultFieldName(MetricDescriptor.Type metricType) {
        switch (metricType) {
            case CUMULATIVE_DOUBLE: 
            case CUMULATIVE_INT64: {
                return "counter";
            }
            case CUMULATIVE_DISTRIBUTION: {
                return "histogram";
            }
        }
        return "value";
    }

    private String removeCommonPrefix(String str, String prefixStr) {
        int commonLen;
        int limit = Math.min(str.length(), prefixStr.length());
        for (commonLen = 0; commonLen < limit && str.charAt(commonLen) == prefixStr.charAt(commonLen); ++commonLen) {
        }
        return str.substring(commonLen);
    }

    private String sanitizeName(String name) {
        return name.replaceAll("^[^a-zA-Z0-9]+|[^a-zA-Z0-9]+$", "").replaceAll("[^a-zA-Z0-9]+", "_").toLowerCase();
    }

    private Stream<org.influxdb.dto.Point> toInfluxPoints(Metric m, String measurementName, String fieldName) {
        return m.getTimeSeriesList().stream().flatMap(timeSeries -> {
            Map<String, String> tags = this.getTags(m.getMetricDescriptor().getLabelKeys(), timeSeries.getLabelValues());
            return timeSeries.getPoints().stream().flatMap(pt -> ((Stream)pt.getValue().match(doubleVal -> Stream.of(org.influxdb.dto.Point.measurement((String)measurementName).addField(fieldName, (Number)doubleVal)), longVal -> Stream.of(org.influxdb.dto.Point.measurement((String)measurementName).addField(fieldName, (Number)longVal)), distribution -> this.getDistributionPoints((Distribution)distribution, measurementName, fieldName), null, null)).map(ptb -> ptb.time(this.getPointMillis((Point)pt), TimeUnit.MILLISECONDS)).map(ptb -> ptb.tag(tags)).map(Point.Builder::build));
        });
    }

    private Stream<Point.Builder> getDistributionPoints(Distribution distr, String measurementName, String fieldName) {
        String prefix = fieldName.isEmpty() ? "" : fieldName + "_";
        ArrayList<Point.Builder> results = new ArrayList<Point.Builder>();
        results.add(org.influxdb.dto.Point.measurement((String)measurementName).addField(prefix + "sum", distr.getSum()).addField(prefix + "count", distr.getCount()));
        List bucketBoundaries = (List)distr.getBucketOptions().match(Distribution.BucketOptions.ExplicitOptions::getBucketBoundaries, noOp -> Collections.emptyList());
        for (int i = 0; i < distr.getBuckets().size(); ++i) {
            Distribution.Bucket bucket = (Distribution.Bucket)distr.getBuckets().get(i);
            String lowerLimit = i > 0 ? "(" + bucketBoundaries.get(i - 1) : "(-Inf";
            String upperLimit = i < bucketBoundaries.size() ? bucketBoundaries.get(i) + "]" : "+Inf)";
            String interval = lowerLimit + "," + upperLimit;
            results.add(org.influxdb.dto.Point.measurement((String)measurementName).addField(prefix + "bucket", bucket.getCount()).tag("bucket", interval));
        }
        return results.stream();
    }

    Map<String, String> getTags(List<LabelKey> labelKeys, List<LabelValue> labelValues) {
        Iterator<LabelKey> keys = labelKeys.iterator();
        Iterator<LabelValue> values = labelValues.iterator();
        HashMap<String, String> result = new HashMap<String, String>();
        while (keys.hasNext() && values.hasNext()) {
            String value = values.next().getValue();
            String key = keys.next().getKey();
            if (value == null || key == null) continue;
            result.put(key, value);
        }
        return result;
    }

    long getPointMillis(Point pt) {
        Timestamp timestamp = pt.getTimestamp();
        return (long)(timestamp.getNanos() / 1000 / 1000) + timestamp.getSeconds() * 1000L;
    }

    private synchronized void connectAndCreateDatabase() {
        try {
            if (this.influx == null) {
                QueryResult query;
                String error;
                this.influx = this.user == null || this.password == null ? InfluxDBFactory.connect((String)this.url) : InfluxDBFactory.connect((String)this.url, (String)this.user, (String)this.password);
                if (this.createDatabase && (error = (query = this.influx.query(new Query("CREATE DATABASE " + this.database))).getError()) != null) {
                    log.error("Error creating database: {}", (Object)error);
                }
            }
        }
        catch (Throwable t) {
            this.influx = null;
            log.error("Could not connect to influx", t);
        }
    }

    @Override
    public synchronized void close() {
        if (this.influx != null) {
            this.influx.close();
            this.influx = null;
        }
    }

    public static InfluxExporterBuilder builder() {
        return new InfluxExporterBuilder();
    }

    public static class InfluxExporterBuilder {
        private String url;
        private String user;
        private String password;
        private String database;
        private String retention;
        private boolean createDatabase;

        InfluxExporterBuilder() {
        }

        public InfluxExporterBuilder url(String url) {
            this.url = url;
            return this;
        }

        public InfluxExporterBuilder user(String user) {
            this.user = user;
            return this;
        }

        public InfluxExporterBuilder password(String password) {
            this.password = password;
            return this;
        }

        public InfluxExporterBuilder database(String database) {
            this.database = database;
            return this;
        }

        public InfluxExporterBuilder retention(String retention) {
            this.retention = retention;
            return this;
        }

        public InfluxExporterBuilder createDatabase(boolean createDatabase) {
            this.createDatabase = createDatabase;
            return this;
        }

        public InfluxExporter build() {
            return new InfluxExporter(this.url, this.user, this.password, this.database, this.retention, this.createDatabase);
        }

        public String toString() {
            return "InfluxExporter.InfluxExporterBuilder(url=" + this.url + ", user=" + this.user + ", password=" + this.password + ", database=" + this.database + ", retention=" + this.retention + ", createDatabase=" + this.createDatabase + ")";
        }
    }
}

