/*
 * Decompiled with CFR 0.152.
 */
package rocks.inspectit.opencensus.influx;

import io.opencensus.metrics.Metrics;
import io.opencensus.metrics.export.Distribution;
import io.opencensus.metrics.export.Metric;
import io.opencensus.metrics.export.MetricDescriptor;
import io.opencensus.metrics.export.MetricProducer;
import io.opencensus.metrics.export.Point;
import io.opencensus.metrics.export.Value;
import io.opencensus.stats.Stats;
import io.opencensus.stats.View;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;
import org.influxdb.dto.BatchPoints;
import org.influxdb.dto.Point;
import org.influxdb.dto.Query;
import org.influxdb.dto.QueryResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import rocks.inspectit.opencensus.influx.InfluxUtils;
import rocks.inspectit.opencensus.influx.StatisticsCache;

public class InfluxExporter
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(InfluxExporter.class);
    private static final Number LONG_ZERO = 0L;
    private static final Number DOUBLE_ZERO = 0.0;
    private final String url;
    private final String user;
    private final String password;
    private final String database;
    private final String retention;
    private boolean createDatabase;
    private boolean exportDifference;
    private Function<String, String> measurementNameProvider;
    private InfluxDB influx;
    private StatisticsCache statsCache;
    private Supplier<Set<MetricProducer>> metricProducerSupplier;
    private Supplier<Set<View>> viewSupplier;

    public InfluxExporter(String url, String user, String password, String database, String retention, boolean createDatabase, boolean exportDifference, Function<String, String> measurementNameProvider) {
        this.url = url;
        this.user = user;
        this.password = password;
        this.database = database;
        this.retention = retention;
        this.createDatabase = createDatabase;
        this.exportDifference = exportDifference;
        this.measurementNameProvider = measurementNameProvider;
        this.metricProducerSupplier = () -> Metrics.getExportComponent().getMetricProducerManager().getAllMetricProducer();
        this.viewSupplier = () -> Stats.getViewManager().getAllExportedViews();
        if (exportDifference) {
            this.statsCache = new StatisticsCache();
            this.export(true);
        }
    }

    public synchronized void export() {
        this.export(false);
    }

    private synchronized void export(boolean dryRun) {
        List<Metric> metrics = this.metricProducerSupplier.get().stream().flatMap(metricProducer -> metricProducer.getMetrics().stream()).collect(Collectors.toList());
        this.export(metrics, dryRun);
    }

    private synchronized void export(Collection<Metric> metrics, boolean dryRun) {
        if (metrics.size() <= 0) {
            return;
        }
        Map<String, View> viewsMap = this.viewSupplier.get().stream().collect(Collectors.toMap(view -> view.getName().asString(), view -> view));
        if (!dryRun) {
            this.connectAndCreateDatabase();
        }
        if (this.influx != null || dryRun) {
            try {
                BatchPoints batch = BatchPoints.database((String)this.database).retentionPolicy(this.retention).build();
                metrics.stream().flatMap(metric -> {
                    String metricName = metric.getMetricDescriptor().getName();
                    String measurementName = this.getMeasurementName(viewsMap, metricName);
                    String fieldName = InfluxUtils.getRawFieldName(metric.getMetricDescriptor().getType(), metricName, measurementName);
                    return this.toInfluxPoints((Metric)metric, InfluxUtils.sanitizeName(measurementName), InfluxUtils.sanitizeName(fieldName));
                }).filter(Objects::nonNull).forEach(arg_0 -> ((BatchPoints)batch).point(arg_0));
                if (!dryRun) {
                    this.influx.write(batch);
                }
            }
            catch (Exception e) {
                log.error("Error writing to InfluxDB", (Throwable)e);
            }
        }
    }

    private String getMeasurementName(Map<String, View> viewsMap, String metricName) {
        String measurementName = null;
        if (this.measurementNameProvider != null) {
            measurementName = this.measurementNameProvider.apply(metricName);
        }
        if (measurementName == null) {
            View view = viewsMap.get(metricName);
            measurementName = InfluxUtils.getRawMeasurementName(metricName, view);
        }
        return measurementName;
    }

    private Stream<org.influxdb.dto.Point> toInfluxPoints(Metric metric, String measurementName, String fieldName) {
        return metric.getTimeSeriesList().stream().flatMap(timeSeries -> {
            Map<String, String> tags = InfluxUtils.createTagMaps(metric.getMetricDescriptor().getLabelKeys(), timeSeries.getLabelValues());
            return timeSeries.getPoints().stream().flatMap(point -> this.toInfluxPoint((Point)point, metric, measurementName, fieldName, tags));
        });
    }

    private Stream<org.influxdb.dto.Point> toInfluxPoint(Point point, Metric metric, String measurementName, String fieldName, Map<String, String> tags) {
        long pointTime = InfluxUtils.getTimestampOfPoint(point);
        Value pointValue = point.getValue();
        Stream builderStream = (Stream)pointValue.match(doubleValue -> this.transformValue(metric, measurementName, fieldName, tags, (Number)doubleValue), longValue -> this.transformValue(metric, measurementName, fieldName, tags, (Number)longValue), distributionValue -> this.getDistributionPoints((Distribution)distributionValue, measurementName, fieldName, tags), null, null);
        return builderStream.map(builder -> builder.time(pointTime, TimeUnit.MILLISECONDS)).map(Point.Builder::build);
    }

    private Stream<Point.Builder> transformValue(Metric metric, String measurementName, String fieldName, Map<String, String> tags, Number value) {
        MetricDescriptor.Type type = metric.getMetricDescriptor().getType();
        boolean isGauge = type == MetricDescriptor.Type.GAUGE_DOUBLE || type == MetricDescriptor.Type.GAUGE_INT64 || type == MetricDescriptor.Type.GAUGE_DISTRIBUTION;
        Optional<Point.Builder> pointBuilder = this.createPointBuilder(measurementName, fieldName, tags, value, isGauge);
        return pointBuilder.map(Stream::of).orElseGet(Stream::empty);
    }

    private Optional<Point.Builder> createPointBuilder(String measurementName, String fieldName, Map<String, String> tags, Number value) {
        return this.createPointBuilder(measurementName, fieldName, tags, value, false);
    }

    private Optional<Point.Builder> createPointBuilder(String measurementName, String fieldName, Map<String, String> tags, Number value, boolean isGauge) {
        Optional<Number> processedValue = this.processValue(isGauge, measurementName, fieldName, tags, value);
        return processedValue.map(number -> org.influxdb.dto.Point.measurement((String)measurementName).addField(fieldName, number).tag(tags));
    }

    private Stream<Point.Builder> getDistributionPoints(Distribution distribution, String measurementName, String fieldName, Map<String, String> tags) {
        String prefix = fieldName.isEmpty() ? "" : fieldName + "_";
        ArrayList<Optional<Point.Builder>> results = new ArrayList<Optional<Point.Builder>>();
        String countFieldName = prefix + "count";
        String sumFieldName = prefix + "sum";
        results.add(this.createPointBuilder(measurementName, countFieldName, tags, distribution.getCount()));
        results.add(this.createPointBuilder(measurementName, sumFieldName, tags, distribution.getSum()));
        HashMap<String, String> tempTagMap = new HashMap<String, String>(tags);
        List bucketBoundaries = (List)distribution.getBucketOptions().match(Distribution.BucketOptions.ExplicitOptions::getBucketBoundaries, noOp -> Collections.emptyList());
        for (int i = 0; i < distribution.getBuckets().size(); ++i) {
            Distribution.Bucket bucket = (Distribution.Bucket)distribution.getBuckets().get(i);
            String lowerLimit = i > 0 ? "(" + bucketBoundaries.get(i - 1) : "(-Inf";
            String upperLimit = i < bucketBoundaries.size() ? bucketBoundaries.get(i) + "]" : "+Inf)";
            String interval = lowerLimit + "," + upperLimit;
            tempTagMap.put("bucket", interval);
            results.add(this.createPointBuilder(measurementName, prefix + "bucket", tempTagMap, bucket.getCount()));
        }
        return results.stream().filter(Optional::isPresent).map(Optional::get);
    }

    private Optional<Number> processValue(boolean isGauge, String measurementName, String fieldName, Map<String, String> tags, Number value) {
        if (this.exportDifference && !isGauge) {
            Number difference = this.statsCache.getDifference(measurementName, fieldName, tags, value);
            if (difference.doubleValue() != 0.0) {
                return Optional.of(difference);
            }
            return Optional.empty();
        }
        return Optional.of(value);
    }

    private synchronized void connectAndCreateDatabase() {
        block5: {
            try {
                if (this.influx != null) break block5;
                this.influx = this.user == null || this.password == null ? InfluxDBFactory.connect((String)this.url) : InfluxDBFactory.connect((String)this.url, (String)this.user, (String)this.password);
                if (!this.createDatabase) break block5;
                try {
                    QueryResult query = this.influx.query(new Query("CREATE DATABASE " + this.database));
                    String error = query.getError();
                    if (error != null) {
                        log.error("Error creating database: {}", (Object)error);
                    }
                }
                catch (Exception e) {
                    log.error("Error creating database: {}", (Throwable)e);
                }
            }
            catch (Throwable t) {
                this.influx = null;
                log.error("Could not connect to InfluxDB", t);
            }
        }
    }

    @Override
    public synchronized void close() {
        if (this.influx != null) {
            this.influx.close();
            this.influx = null;
        }
    }

    public static InfluxExporterBuilder builder() {
        return new InfluxExporterBuilder();
    }

    void setInflux(InfluxDB influx) {
        this.influx = influx;
    }

    void setMetricProducerSupplier(Supplier<Set<MetricProducer>> metricProducerSupplier) {
        this.metricProducerSupplier = metricProducerSupplier;
    }

    void setViewSupplier(Supplier<Set<View>> viewSupplier) {
        this.viewSupplier = viewSupplier;
    }

    public static class InfluxExporterBuilder {
        private String url;
        private String user;
        private String password;
        private String database;
        private String retention;
        private boolean createDatabase;
        private boolean exportDifference;
        private Function<String, String> measurementNameProvider;

        InfluxExporterBuilder() {
        }

        public InfluxExporterBuilder url(String url) {
            this.url = url;
            return this;
        }

        public InfluxExporterBuilder user(String user) {
            this.user = user;
            return this;
        }

        public InfluxExporterBuilder password(String password) {
            this.password = password;
            return this;
        }

        public InfluxExporterBuilder database(String database) {
            this.database = database;
            return this;
        }

        public InfluxExporterBuilder retention(String retention) {
            this.retention = retention;
            return this;
        }

        public InfluxExporterBuilder createDatabase(boolean createDatabase) {
            this.createDatabase = createDatabase;
            return this;
        }

        public InfluxExporterBuilder exportDifference(boolean exportDifference) {
            this.exportDifference = exportDifference;
            return this;
        }

        public InfluxExporterBuilder measurementNameProvider(Function<String, String> measurementNameProvider) {
            this.measurementNameProvider = measurementNameProvider;
            return this;
        }

        public InfluxExporter build() {
            return new InfluxExporter(this.url, this.user, this.password, this.database, this.retention, this.createDatabase, this.exportDifference, this.measurementNameProvider);
        }

        public String toString() {
            return "InfluxExporter.InfluxExporterBuilder(url=" + this.url + ", user=" + this.user + ", password=" + this.password + ", database=" + this.database + ", retention=" + this.retention + ", createDatabase=" + this.createDatabase + ", exportDifference=" + this.exportDifference + ", measurementNameProvider=" + this.measurementNameProvider + ")";
        }
    }
}

