/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2016 Christian Schudt
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package rocks.xmpp.extensions.geoloc;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.function.Consumer;

import rocks.xmpp.core.session.Manager;
import rocks.xmpp.core.session.XmppSession;
import rocks.xmpp.core.stanza.MessageEvent;
import rocks.xmpp.core.stanza.model.Message;
import rocks.xmpp.extensions.geoloc.model.GeoLocation;
import rocks.xmpp.extensions.pubsub.PubSubManager;
import rocks.xmpp.extensions.pubsub.PubSubService;
import rocks.xmpp.extensions.pubsub.model.Item;
import rocks.xmpp.extensions.pubsub.model.event.Event;
import rocks.xmpp.util.XmppUtils;
import rocks.xmpp.util.concurrent.AsyncResult;

/**
 * Manages the publishing of user location and the notification of it.
 *
 * @author Christian Schudt
 * @see <a href="https://xmpp.org/extensions/xep-0080.html">XEP-0080: User Location</a>
 */
public final class GeoLocationManager extends Manager {

    private final Set<Consumer<GeoLocationEvent>> geoLocationListeners = new CopyOnWriteArraySet<>();

    private final Consumer<MessageEvent> messageListener;

    private GeoLocationManager(XmppSession xmppSession) {
        super(xmppSession, true);

        messageListener = e -> {
            Message message = e.getMessage();
            Event event = message.getExtension(Event.class);
            if (event != null) {
                for (Item item : event.getItems()) {
                    Object payload = item.getPayload();
                    if (payload instanceof GeoLocation) {
                        // Notify the listeners about the reception.
                        XmppUtils.notifyEventListeners(geoLocationListeners,
                                new GeoLocationEvent(GeoLocationManager.this, (GeoLocation) payload,
                                        message.getFrom()));
                    }
                }
            }
        };
    }

    @Override
    protected void onEnable() {
        super.onEnable();
        xmppSession.addInboundMessageListener(messageListener);
    }

    @Override
    protected void onDisable() {
        super.onDisable();
        xmppSession.removeInboundMessageListener(messageListener);
    }

    /**
     * Publishes a geo location to the personal eventing service.
     *
     * @param geoLocation The geo location.
     * @return The async result with the item id, generated by the pubsub service.
     */
    public AsyncResult<String> publish(GeoLocation geoLocation) {
        PubSubService pepService = xmppSession.getManager(PubSubManager.class).createPersonalEventingService();
        return pepService.node(GeoLocation.NAMESPACE).publish(geoLocation);
    }

    /**
     * Adds a listener, which allows to listen for geo location changes.
     *
     * @param geoLocationListener The listener.
     * @see #removeGeoLocationListener(Consumer)
     */
    public void addGeoLocationListener(Consumer<GeoLocationEvent> geoLocationListener) {
        geoLocationListeners.add(geoLocationListener);
    }

    /**
     * Removes a previously added geo location listener.
     *
     * @param geoLocationListener The listener.
     * @see #addGeoLocationListener(Consumer)
     */
    public void removeGeoLocationListener(Consumer<GeoLocationEvent> geoLocationListener) {
        geoLocationListeners.remove(geoLocationListener);
    }

    @Override
    protected void dispose() {
        geoLocationListeners.clear();
    }
}
