/*
 * Copyright Бездна (c) 2018.
 */
package ru.abyss.settings.importer;

import java.io.File;
import java.io.RandomAccessFile;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Properties;
import java.util.Set;

import org.h2.Driver;
import org.reflections.Reflections;

import net.sf.sevenzipjbinding.IInArchive;
import net.sf.sevenzipjbinding.SevenZip;
import net.sf.sevenzipjbinding.impl.RandomAccessFileInStream;
import net.sf.sevenzipjbinding.simple.ISimpleInArchiveItem;

import ru.abyss.settings.AbyssVersion;
import ru.abyss.settings.ProgressCallback;

/**
 * @author Minu <<a href="minu-moto@mail.ru">minu-moto@mail.ru</a>>
 * @since 01.06.2018 12:27:46
 */
public class DBImporter {

//	private static Logger logger = LoggerFactory.getLogger(DBImporter.class);

	static {
		Driver.load();
	}

	public static AbyssVersion getAbyssVersion(File archive) throws Exception {
		try (RandomAccessFileInStream inputFile = new RandomAccessFileInStream(new RandomAccessFile(archive, "r"));
				IInArchive arch = SevenZip.openInArchive(null, inputFile)) {
			boolean sb = false;
			boolean st = false;
			boolean tmp = false;
			boolean set = false;
			boolean h2 = false;
			for (ISimpleInArchiveItem item : arch.getSimpleInterface().getArchiveItems()) {
				String path = item.getPath();
				if (path.matches("(?iu)^(server(/|\\\\))?servbase.mdb$"))
					sb = true;
				if (path.matches("(?iu)^(server(/|\\\\))?stock.mdb$"))
					st = true;
				if (path.matches("(?iu)^(server(/|\\\\))?template.mdb$"))
					tmp = true;
				if (path.matches("(?iu)^(server(/|\\\\))?settings.mdb$"))
					set = true;
				if (path.contains("abyss.mv.db"))
					h2 = true;
			}
			if (h2)
				return AbyssVersion.V6_1;
			else if (sb && st && tmp && set)
				return AbyssVersion.V5_7;
			else
				return null;
		}
	}

	private static long[] getDBVersion(Connection h2Connection) throws Exception {
		try (Statement st = h2Connection.createStatement()) {
			st.execute("select * from version");
			try (ResultSet rs = st.getResultSet()) {
				if (rs.next())
					return new long[] {rs.getLong("num"), rs.getLong("rev"), rs.getLong("tfactory_id")};
				throw new ImportException("Не удалось определить версию БД");
			}
		}
	}

	public static boolean checkImporterVersion(Class<?> clazz, String version) {
		ImporterVersion[] vs = clazz.getAnnotationsByType(ImporterVersion.class);
		if (vs != null)
			for (ImporterVersion v : vs)
				if (v.value().equalsIgnoreCase(version))
					return true;
		return false;
	}

	public static void doImport(Path h2DB, Connection pgConnection, Properties props, Long factoryId, Long userId, boolean isLoadPrices,
			boolean isLoadConsistRemains, boolean isLoadContractors, ProgressCallback callback) throws Exception {
		try (Connection h2 = DriverManager.getConnection("jdbc:h2:" + h2DB + "/abyss;TRACE_LEVEL_FILE=4",
				props.getProperty("export.h2.login"), props.getProperty("export.h2.password"))) {
			h2.setAutoCommit(false);

			Importer importer = null;
			long[] version = getDBVersion(h2);
			String ver = version[0] + "." + version[1];
			Reflections refl = new Reflections(DBImporter.class.getPackage().getName());
			Set<Class<?>> importers = refl.getTypesAnnotatedWith(ImporterVersion.class);
			if (importers != null) {
				Set<Class<?>> importers2 = refl.getTypesAnnotatedWith(ImporterVersions.class);
				if (importers2 != null)
					importers.addAll(importers2);
				for (Class<?> clazz : importers)
					if (Importer.class.isAssignableFrom(clazz) && checkImporterVersion(clazz, ver))
						importer = (Importer) clazz.newInstance();
			}
			if (importer == null)
				throw new ImportException("Версия БД v" + ver + " - не поддерживается системой");
			else if (callback != null)
				callback.setProgress("Версия БД: " + ver, 0, -1);

			importer.doImport(h2, pgConnection, factoryId, userId, factoryId.equals(version[2]), isLoadPrices, isLoadConsistRemains, isLoadContractors, callback);
		}
	}

}