/*
 * Decompiled with CFR 0.152.
 */
package ru.dlabs71.library.email.client.receiver;

import jakarta.mail.Flags;
import jakarta.mail.Folder;
import jakarta.mail.Message;
import jakarta.mail.MessagingException;
import jakarta.mail.Session;
import java.security.GeneralSecurityException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.angus.mail.imap.IMAPStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.dlabs71.library.email.client.receiver.ReceiverDClient;
import ru.dlabs71.library.email.converter.incoming.BaseMessageConverter;
import ru.dlabs71.library.email.converter.incoming.MessageViewConverter;
import ru.dlabs71.library.email.dto.message.common.EmailParticipant;
import ru.dlabs71.library.email.dto.message.incoming.DefaultIncomingMessage;
import ru.dlabs71.library.email.dto.message.incoming.IncomingMessage;
import ru.dlabs71.library.email.dto.message.incoming.MessageView;
import ru.dlabs71.library.email.dto.pageable.PageRequest;
import ru.dlabs71.library.email.exception.FolderOperationException;
import ru.dlabs71.library.email.exception.SessionException;
import ru.dlabs71.library.email.property.ImapProperties;
import ru.dlabs71.library.email.property.SessionPropertyCollector;
import ru.dlabs71.library.email.type.Protocol;
import ru.dlabs71.library.email.util.JavaCoreUtils;
import ru.dlabs71.library.email.util.RetryableUtils;

public class IMAPDClient
implements ReceiverDClient {
    private static final Logger log = LoggerFactory.getLogger(IMAPDClient.class);
    private static final Protocol PROTOCOL = Protocol.IMAP;
    public static final String DEFAULT_INBOX_FOLDER_NAME = "INBOX";
    public static final String DEFAULT_OUTBOX_FOLDER_NAME = "OUTBOX";
    private final Session session;
    private final Properties properties;
    private final IMAPStore store;
    private final EmailParticipant principal;
    private final int maxAttemptsOfRequest;
    private final int attemptDelayOfRequest;

    public IMAPDClient(ImapProperties imapProperties) {
        JavaCoreUtils.notNullArgument(imapProperties, "imapProperties");
        this.principal = EmailParticipant.of(imapProperties.getEmail());
        this.maxAttemptsOfRequest = imapProperties.getMaxAttemptsOfRequest();
        this.attemptDelayOfRequest = imapProperties.getAttemptDelayOfRequest();
        log.debug("Principal object were created. {}", (Object)this.principal);
        this.properties = this.collectProperties(imapProperties);
        log.debug("Configuration properties were created");
        this.session = this.connect();
        log.debug("Session was created");
        this.store = IMAPDClient.createStore(this.session, imapProperties.getEmail(), imapProperties.getPassword());
        log.debug("Store was created. Client is ready to receiving messages!");
    }

    private Properties collectProperties(ImapProperties imapProperties) {
        Properties props;
        try {
            props = SessionPropertyCollector.createCommonProperties(imapProperties, PROTOCOL);
        }
        catch (GeneralSecurityException e) {
            throw new SessionException("The creation of a connection failed because of the following error: " + e.getMessage());
        }
        props.put("mail.imap.partialfetch", (Object)imapProperties.isPartialFetch());
        props.put("mail.imap.fetchsize", (Object)imapProperties.getFetchSize());
        props.put("mail.imap.statuscachetimeout", (Object)imapProperties.getStatusCacheTimeout());
        props.put("mail.imap.appendbuffersize", (Object)imapProperties.getAppendBufferSize());
        props.put("mail.imap.connectionpoolsize", (Object)imapProperties.getConnectionPoolSize());
        props.put("mail.imap.connectionpooltimeout", (Object)imapProperties.getConnectionPoolTimeout());
        return props;
    }

    @Override
    public Session connect() throws SessionException {
        try {
            return Session.getInstance((Properties)this.properties);
        }
        catch (Exception e) {
            throw new SessionException("The creation of a connection failed because of the following error: " + e.getMessage());
        }
    }

    @Override
    public final String getProtocolName() {
        return PROTOCOL.getProtocolName();
    }

    @Override
    public final EmailParticipant getPrincipal() {
        return this.principal;
    }

    private static IMAPStore createStore(Session session, String email, String password) {
        try {
            log.info("Connect to mailbox: " + email);
            IMAPStore store = (IMAPStore)session.getStore(PROTOCOL.getProtocolName());
            store.connect(email, password);
            return store;
        }
        catch (MessagingException e) {
            throw new SessionException("Creating session finished with the error: " + e.getMessage(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Integer getTotalCount(String folderName) {
        Folder folder = this.openFolderForRead(folderName);
        try {
            Integer n = this.getTotalCount(folder);
            return n;
        }
        finally {
            this.closeFolder(folder);
        }
    }

    private Integer getTotalCount(Folder folder) throws FolderOperationException {
        log.debug("Gets total messages from the folder {}", (Object)folder);
        try {
            return folder.getMessageCount();
        }
        catch (MessagingException e) {
            throw new FolderOperationException("Getting a count of messages in the folder with the name " + folder.getName() + " finished with the error: " + e.getMessage(), e);
        }
    }

    @Override
    public List<MessageView> checkEmailMessages(String folderName, PageRequest pageRequest) {
        log.debug("Checks email messages from the folder {} and page request is {}", (Object)folderName, (Object)pageRequest);
        Folder folder = this.openFolderForRead(folderName);
        Stream<Message> messages = this.getMessages(folder, pageRequest);
        List<MessageView> result = messages.map(MessageViewConverter::convert).collect(Collectors.toList());
        log.debug(result.size() + " email messages was got");
        this.closeFolder(folder);
        return result;
    }

    @Override
    public List<IncomingMessage> readMessages(String folderName, PageRequest pageRequest) {
        log.debug("Reads email messages from the folder {} and page request is {}", (Object)folderName, (Object)pageRequest);
        Folder folder = this.openFolderForWrite(folderName);
        Stream<Message> messages = this.getMessages(folder, pageRequest);
        List<IncomingMessage> result = messages.map(BaseMessageConverter::convertToIncomingMessage).collect(Collectors.toList());
        log.debug(result.size() + " email messages was got");
        this.closeFolder(folder);
        return result;
    }

    @Override
    public IncomingMessage readMessageById(String folderName, int id) {
        Message message;
        log.debug("Reads one message by id = {} and folder name = {}", (Object)id, (Object)folderName);
        Folder folder = this.openFolderForWrite(folderName);
        try {
            message = RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> folder.getMessage(id));
        }
        catch (MessagingException e) {
            throw new FolderOperationException("Reading the message with id=" + id + " in the folder with name " + folderName + " finished the error: " + e.getMessage(), e);
        }
        DefaultIncomingMessage incomingMessage = BaseMessageConverter.convertToIncomingMessage(message);
        this.closeFolder(folder);
        return incomingMessage;
    }

    @Override
    public Folder openFolderForRead(String folderName) {
        if (folderName == null) {
            folderName = DEFAULT_INBOX_FOLDER_NAME;
        }
        return this.openFolder(folderName, 1);
    }

    @Override
    public Folder openFolderForWrite(String folderName) {
        if (folderName == null) {
            folderName = DEFAULT_INBOX_FOLDER_NAME;
        }
        return this.openFolder(folderName, 2);
    }

    private Folder openFolder(String folderName, int mode) {
        Folder folder;
        log.debug("Try to open folder {} with access mode is {}", (Object)folderName, (Object)mode);
        if (this.store == null) {
            throw new FolderOperationException("The store is null");
        }
        try {
            folder = RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> this.store.getFolder(folderName));
            RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> folder.open(mode));
            log.debug("Folder is opened");
        }
        catch (MessagingException e) {
            throw new FolderOperationException("The folder with the name " + folderName + " couldn't be opened because of the following error: " + e.getMessage(), e);
        }
        return folder;
    }

    @Override
    public void closeFolder(Folder folder) {
        log.debug("Try to close the folder {}", (Object)folder);
        if (folder == null) {
            return;
        }
        try {
            RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> folder.close());
            log.debug("Folder is closed.");
        }
        catch (MessagingException e) {
            log.warn("The folder with the name " + folder.getName() + " couldn't be closed because of the following error: " + e.getMessage());
        }
    }

    @Override
    public boolean deleteMessage(String folderName, int id) {
        log.debug("Deletes one message by id = {} and folder name = {}", (Object)id, (Object)folderName);
        Folder folder = this.openFolderForWrite(folderName);
        try {
            RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> {
                Message message = folder.getMessage(id);
                message.setFlag(Flags.Flag.DELETED, true);
            });
        }
        catch (MessagingException e) {
            log.warn("The message with id={} wasn't marked as deleted because of the following error: {}", (Object)id, (Object)e.getMessage());
            this.closeFolder(folder);
            return false;
        }
        try {
            folder.expunge();
        }
        catch (MessagingException e) {
            throw new FolderOperationException("The folder with the name " + folder.getName() + " couldn't be expunge because of the following error: " + e.getMessage());
        }
        finally {
            this.closeFolder(folder);
        }
        return true;
    }

    @Override
    public Map<Integer, Boolean> deleteMessages(String folderName, Collection<Integer> ids) {
        log.debug("Deletes one message from the folder {} by the message ids = {}", (Object)folderName, ids);
        Folder folder = this.openFolderForWrite(folderName);
        HashMap<Integer, Boolean> result = new HashMap<Integer, Boolean>();
        ids.forEach(id -> {
            try {
                RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> {
                    Message message = folder.getMessage(id.intValue());
                    message.setFlag(Flags.Flag.DELETED, true);
                });
                result.put((Integer)id, true);
            }
            catch (MessagingException e) {
                log.warn("The message with id=" + id + " wasn't marked as deleted because of the following error: " + e.getMessage());
                result.put((Integer)id, false);
            }
        });
        try {
            RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> ((Folder)folder).expunge());
        }
        catch (MessagingException e) {
            throw new FolderOperationException("The folder with the name " + folder.getName() + " couldn't be expunge because of the following error: " + e.getMessage());
        }
        this.closeFolder(folder);
        return result;
    }

    @Override
    public Map<Integer, Boolean> deleteAllMessages(String folderName) {
        log.debug("Deletes all the messages from the folder {}", (Object)folderName);
        Folder folder = this.openFolderForWrite(folderName);
        HashMap<Integer, Boolean> result = new HashMap<Integer, Boolean>();
        this.getMessages(folder, PageRequest.of(0, Integer.MAX_VALUE)).forEach(message -> {
            try {
                RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> message.setFlag(Flags.Flag.DELETED, true));
                result.put(message.getMessageNumber(), true);
            }
            catch (MessagingException e) {
                log.warn("The message with id=" + message.getMessageNumber() + " wasn't marked as deleted because of the following error: " + e.getMessage());
                result.put(message.getMessageNumber(), false);
            }
        });
        try {
            RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> ((Folder)folder).expunge());
        }
        catch (MessagingException e) {
            throw new FolderOperationException("The folder with the name " + folder.getName() + " couldn't be expunge because of the following error: " + e.getMessage());
        }
        this.closeFolder(folder);
        return result;
    }

    private Stream<Message> getMessages(Folder folder, PageRequest pageRequest) {
        int totalCount = this.getTotalCount(folder);
        int end = Math.min(totalCount, pageRequest.getEnd() + 1);
        log.debug("Gets message from folder {} with page request {}. The folder has {} messages", new Object[]{folder, pageRequest, totalCount});
        try {
            return RetryableUtils.retry(this.maxAttemptsOfRequest, (long)this.attemptDelayOfRequest, () -> Arrays.stream(folder.getMessages(pageRequest.getStart() + 1, end)));
        }
        catch (MessagingException e) {
            throw new FolderOperationException("The get list message operation has failed: " + e.getMessage());
        }
    }
}

