/*
 * Author 2018 Alexey Kudrin.
 */
package ru.ilb.adjustdate.utils;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.time.LocalDate;
import javax.inject.Named;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import org.joda.time.DateTime;
import org.joda.time.MutableDateTime;
import org.xml.sax.InputSource;
import ru.ilb.meta.common.daysadjustment.Daysadjustment;
import ru.ilb.util.date.daysadjustment.DaysadjustmentImpl;

/**
 * Для работы нужно:
 *
 * <pre>
 *  &lt;resource-env-ref&gt;
 *   &lt;description>Сервисы meta&lt;/description&gt;
 *   &lt;resource-env-ref-name>ru.bystrobank.apps.meta.url&lt;/resource-env-ref-name&gt;
 *   &lt;resource-env-ref-type>javax.jms.Queue&lt;/resource-env-ref-type&gt;
 *  &lt;/resource-env-ref&gt;
 * </pre>
 *
 * @author Alexey Kudrin
 */
@Named
public class AdjustDate {

    private final URI daysadjustmentURI;

    public AdjustDate(URI daysadjustmentURI) {
        this.daysadjustmentURI = daysadjustmentURI;
    }

//    public AdjustDate(String daysadjustmentURIString) throws URISyntaxException {
//        this.daysadjustmentURI = new URI(daysadjustmentURIString);
//    }
    /**
     * Получение ближайшего рабочего дня.<ul>
     * <li>Если дата приходится на выходной, праздничный или нежелательный день,
     * то она заменяется на ближайший последующий рабочий день текущего
     * месяца;</li>
     * <li>Если таковой отсутствует, то она заменяется на ближайший предыдущий
     * рабочий день текущего месяца;</li>
     * </ul>
     *
     * @param localDate Преобразуемая дата.
     * @param sameMonth В том же месяце (поиск назад, если впереди нет рабочих дней)
     * @return Следующий рабочий день или без изменений.
     */
    public LocalDate adjust(LocalDate localDate, boolean sameMonth) {
        DaysadjustmentImpl adjustmentCalendar = getDaysAdjustmentImpl();

        int month = localDate.getMonthValue();

        MutableDateTime mdt = new DateTime(DateUtilsConverter.asDate(localDate)).toMutableDateTime();

        // находим ближайший последующий рабочий день текущего месяца
        while ((adjustmentCalendar.isWeekend(mdt.toDateTime())
                || adjustmentCalendar.isHoliday(mdt.toDateTime())
                || adjustmentCalendar.isUnwantedday(mdt.toDateTime())) && (!sameMonth || month == mdt.getMonthOfYear())) {
            mdt.addDays(1);
        }

        if (sameMonth && month != mdt.getMonthOfYear()) {
            mdt.addDays(-1);

            // находим ближайший предыдущий рабочий день
            while ((adjustmentCalendar.isWeekend(mdt.toDateTime())
                    || adjustmentCalendar.isHoliday(mdt.toDateTime())
                    || adjustmentCalendar.isUnwantedday(mdt.toDateTime()
                    ) && month == mdt.getMonthOfYear())) {
                mdt.addDays(-1);
            }
        }

//        if (month != mdt.getMonthOfYear()) {
//            throw new RuntimeException("Can't instantiate DaysadjustmentImpl, wrong arguments");
//        }

        return DateUtilsConverter.asLocalDate(mdt.toDate());
    }

    private static DaysadjustmentImpl daysAdjustmentImpl = null;

    private DaysadjustmentImpl getDaysAdjustmentImpl() {
        if (daysAdjustmentImpl == null) {
            Daysadjustment da;

            try {
                da = getDaysadjustment();
            } catch (JAXBException ex) {
                throw new RuntimeException("Can't instantiate DaysadjustmentImpl", ex);
            } catch (IllegalArgumentException | MalformedURLException ex) {
                throw new RuntimeException("Can't instantiate DaysadjustmentImpl, wrong arguments", ex);
            } catch (IOException ex) {
                throw new RuntimeException("Can't instantiate DaysadjustmentImpl, wrong arguments", ex);
            }

            daysAdjustmentImpl = new DaysadjustmentImpl(da);
        }

        return daysAdjustmentImpl;
    }

    private Daysadjustment getDaysadjustment() throws JAXBException, MalformedURLException {
        // По аналогии с:
        // Daysadjustment da = (Daysadjustment) MetaFactory.newInstance(Daysadjustment.class, "data/daysadjustment/daysadjustment_cr.xml");

        JAXBContext jaxbContext = JAXBContext.newInstance(Daysadjustment.class);
        Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();
        URL url= daysadjustmentURI.toURL();

        InputSource is = new InputSource(url.toExternalForm());
        is.setEncoding("UTF-8");
        Daysadjustment da = (Daysadjustment) jaxbUnmarshaller.unmarshal(url);
        return da;
    }
}
