/*
 * DateEntity.java
 *
 * Created on 10 Апрель 2006 г., 14:30
 *
 */
package ru.ilb.common.rfc3339;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import org.joda.time.DateTime;
import org.joda.time.Period;

/**
 * Абстрактный класс с базовыми методами парсинга по rfc3339
 *
 * @author Alexander Soklakov
 * @version 0.1.0
 */
public class DateEntity {

    public static int PRECISION_DAY = 0;
    public static int PRECISION_HOUR_OF_DAY = 1;
    public static int PRECISION_MINUTE = 2;
    public static int PRECISION_SECOND = 3;
    public static int PRECISION_MILLISECOND = 4;

    public static int[][] calendar_precisions = new int[][]{
        new int[]{Calendar.HOUR_OF_DAY, Calendar.MINUTE, Calendar.SECOND, Calendar.MILLISECOND},
        new int[]{Calendar.MINUTE, Calendar.SECOND, Calendar.MILLISECOND},
        new int[]{Calendar.SECOND, Calendar.MILLISECOND},
        new int[]{Calendar.MILLISECOND},
        new int[]{}
    };

    private DateTime START = null;
    private DateTime END = null;
    private Period PERIOD = null;
    private int PRECISION = PRECISION_MILLISECOND;

    /**
     * Creates a new instance of DateEntityImpl
     */
    private DateEntity() {
    }

    /**
     * Creates a new instance of DateEntityImpl
     */
    public DateEntity(Calendar c, Integer precision) {
        if (precision != null) {
            this.PRECISION = precision;
        }
        this.START = new DateTime(round(c));
    }

    /**
     * Creates a new instance of DateEntityImpl
     */
    public DateEntity(String str, Integer precision) throws IllegalArgumentException {
        if (precision != null) {
            this.PRECISION = precision;
        }
        try {
            this.START = round(new DateTime(str));
        } catch (IllegalArgumentException ex1) {
            // Разбор периода
            // строка периода - start/duration/end
            String tmp[] = str.split("/");
            if (tmp.length == 3) {
                this.START = round(new DateTime(tmp[0]));
                this.PERIOD = new Period(tmp[1]);
                this.END = round(new DateTime(tmp[2]));
            } else if (tmp.length == 2) {
                if (!str.startsWith("P")) {
                    this.START = round(new DateTime(tmp[0]));
                    this.PERIOD = new Period(tmp[1]);
                } else {
                    this.PERIOD = new Period(tmp[0]);
                    this.END = round(new DateTime(tmp[1]));
                }
            } else {
                this.PERIOD = new Period(tmp[0]);
            }
        }
    }

    /**
     * Разбор периода
     *
     * @param str строка с периодом
     * @return расширение класса {@link DateEntity}
     * @throws java.lang.IllegalArgumentException если строка не соответсвует формату
     */
    public static DateEntity parse(String str, Integer precision) throws IllegalArgumentException {
        return new DateEntity(str, precision);
    }

    /**
     * @return true - Начальная дата задана
     */
    public boolean haveStart() {
        return this.START != null;
    }

    /**
     * @return true - Период задан
     */
    public boolean havePeriod() {
        return this.PERIOD != null;
    }

    /**
     * @return true - Конечная дата задана
     */
    public boolean haveEnd() {
        return this.END != null;
    }

    /**
     * @return Начальная дата {@link DateTime}
     */
    public DateTime getStart() {
        return this.START;
    }

    /**
     * @return Период {@link Period}
     */
    public Period getPeriod() {
        return this.PERIOD;
    }

    /**
     * @return Конечная дата {@link DateTime}
     */
    public DateTime getEnd() {
        return this.END;
    }

    public int getPrecision() {
        return this.PRECISION;
    }

    @Override
    public String toString() {
        String res = haveStart() ? this.START.toString() : "";
        res += havePeriod() ? (res.equals("") ? "" : "/") + this.PERIOD.toString() : "";
        res += haveEnd() ? (res.equals("") ? "" : "/") + this.END.toString() : "";
        return res;
    }

    /**
     * Получить массив дат из периода. Возвращает значения, в зависимости от того, как задан DateEntity:
     * <li>date или time или datetime - одну дату
     * <li>start/period/end - с даты старта по дату завершения, в соответствии с периодом
     * <li>start/period - с даты старта по limit, в соответствии с периодом
     * <li>period/end - с limit по дату завершения, в соответствии с периодом; периоды отсчитываются от даты завершения
     * <li>period - если limit больше текущей даты, то с текущей даты по limit, в соответствии с периодом; если limit меньше текущей даты, то с limit по текущую дату, в
     * соответствии с периодом, периоды в этом случае отсчитываются от текущей даты
     *
     * @param limit предельная дата, в случае когда не указан один из пределов; если limit == null, то в качестве предельной используется текущая дата
     * @return массив дат
     */
    public Calendar[] getDates(final Calendar limit) {

        // TODO какая точность?
        Calendar curr = round(Calendar.getInstance());
        Calendar lim = limit == null ? (Calendar) curr.clone() : round(limit);

        List<Calendar> dates = new ArrayList();

        if (!havePeriod()) {
            if (haveStart()) {
                dates.add((Calendar) this.START.toGregorianCalendar().clone());
            }
        } else {
            boolean use_next = true;
            Calendar start;
            Calendar end;
            if (haveStart() && haveEnd()) {
                start = (Calendar) this.START.toGregorianCalendar().clone();
                end = (Calendar) this.END.toGregorianCalendar().clone();
            } else if (haveStart() && !haveEnd()) {
                start = (Calendar) this.START.toGregorianCalendar().clone();
                end = lim;
            } else if (!haveStart() && haveEnd()) {
                start = lim;
                end = (Calendar) this.END.toGregorianCalendar().clone();
                use_next = false;
            } else {
                if (lim.after(curr)) {
                    start = curr;
                    end = lim;
                } else {
                    start = lim;
                    end = curr;
                    use_next = false;
                }
            }

            int i = 0;
            if (use_next) {
                Calendar gc = (Calendar) start.clone();
                while (gc.compareTo(end) <= 0) {
                    dates.add(gc);
                    gc = next(start, ++i);
                }
            } else {
                Calendar gc = (Calendar) end.clone();
                while (gc.compareTo(start) >= 0) {
                    dates.add(gc);
                    gc = previous(end, ++i);
                }
            }
        }

        Calendar[] c = dates.toArray(new Calendar[dates.size()]);
        Arrays.sort(c);
        return c;
    }

    /**
     * Проверить принадлежит ли дата диапазону
     *
     * @return true - если да
     */
    public boolean inRange(Calendar date) {
        Calendar[] da = getDates(date);
        // Проверить наличие даты в массиве дат
        for (int i = 0; i < da.length; i++) {
            if (da[i].compareTo(date) == 0) {
                return true;
            }
        }
        return false;
    }

    /**
     * Получить дату через i периодов от date
     */
    public Calendar next(Calendar date, int i) {
        if (this.PERIOD == null) {
            return null;
        }
        Calendar gc = (Calendar) date.clone();
        gc.add(Calendar.MILLISECOND, this.PERIOD.getMillis() * i);
        gc.add(Calendar.SECOND, this.PERIOD.getSeconds() * i);
        gc.add(Calendar.MINUTE, this.PERIOD.getMinutes() * i);
        gc.add(Calendar.HOUR_OF_DAY, this.PERIOD.getHours() * i);
        gc.add(Calendar.DAY_OF_MONTH, this.PERIOD.getDays() * i);
        gc.add(Calendar.MONTH, this.PERIOD.getMonths() * i);
        gc.add(Calendar.YEAR, this.PERIOD.getYears() * i);
        gc.add(Calendar.WEEK_OF_YEAR, this.PERIOD.getWeeks() * i);
        return gc;
    }

    /**
     * Получить дату на i периодов меньше date
     */
    public Calendar previous(Calendar date, int i) {
        return next(date, -i);
    }

    public static Calendar round(Calendar calendar, Integer precision) {
        Calendar c = (Calendar) calendar.clone();
        if (precision != null) {
            for (int i = 0; i < calendar_precisions[precision].length; ++i) {
                c.set(calendar_precisions[precision][i], 0);
            }
        }
        return c;
    }

    private Calendar round(Calendar calendar) {
        return DateEntity.round(calendar, this.PRECISION);
    }

    private DateTime round(DateTime dt) {
        return new DateTime(DateEntity.round(dt.toGregorianCalendar(), this.PRECISION));
    }

}
