/*
 * DateConverter.java
 *
 * Created on 10 Апрель 2006 г., 16:28
 */
package ru.ilb.common.rfc3339.util;

import java.util.Date;
import java.util.GregorianCalendar;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.util.Calendar;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

/**
 * Класс конвертации и преобразования дат
 *
 * @author Nick S. Groznyh
 */
public class DateConverter {

    public static GregorianCalendar gregorian_calendar = new GregorianCalendar();
    public static DatatypeFactory datatype_factory = null;

    /**
     * Создать строку-период
     *
     * @param startDate начальная дата
     * @param duration интервал
     * @param endDate конечная дата
     * @return строка-период
     */
    public static String makePeriod(String startDate, String duration, String endDate) {
        String res = startDate;
        if (!duration.equals("")) {
            if (res.equals("")) {
                res = duration;
            } else {
                res = res + "/" + duration;
            }
        }
        if (!endDate.equals("")) {
            if (res.equals("")) {
                res = endDate;
            } else {
                res = res + "/" + endDate;
            }
        }
        return res;
    }

    /**
     * Получить количество милисекунд от 01-01-1970
     *
     * @param iso_date_time дата в формате iso
     * @return милисекунды от 01-01-1970
     */
    public static String getMsFromDate(String iso_date_time) {
        return "" + new DateTime(iso_date_time).getMillis();
    }

    /**
     * Получить время в милисикундах
     *
     * @param time время
     * @return милисекунды
     */
    public static String getMsFromTime(String time) {
        return "" + new DateTime(time).getMillis();
    }

    /**
     * Получить интервал в милисекундах от текущей даты
     *
     * @param period интервал
     * @return милисекунды
     */
    public static String getMsFromInterval(String period) {
        Interval i = new Interval(new DateTime(Calendar.getInstance()), new Period(period));
        return "" + (i.getEndMillis() - i.getStartMillis());
    }

    /**
     * Получить дату в формате iso из милисекунд
     *
     * @param ms милисекунды
     * @return дата в формате iso
     */
    public static String getDateString(long ms) {
        return new DateTime(ms).toString();
    }

    /**
     * Получить дату в формате iso из милисекунд
     *
     * @param ms милисекунды
     * @return дата в формате iso
     */
    public static String getDateString(String ms) {
        return getDateString(Long.parseLong(ms));
    }

    /**
     * Получить дату в формате iso из Calendar
     *
     * @param calendar
     * @return дата в формате iso
     */
    public static String getDateString(Calendar calendar) {
        return new DateTime(calendar).toString();
    }

    /**
     * Получить дату в формате iso из Calendar без части времени
     *
     * @param calendar
     * @return дата в формате iso
     */
    public static String getDateOnlyString(Calendar calendar) {
        DateTime dt = new DateTime(calendar);
        DateTimeFormatter fmt = ISODateTimeFormat.date();
        return fmt.print(dt);
    }

    /**
     * Получить время из милисекунд
     *
     * @param ms милисекунды
     * @return время
     */
    public static String getTimeString(long ms) {
        DateTimeFormatterBuilder fb = new DateTimeFormatterBuilder()
                .appendHourOfDay(2).appendLiteral(":")
                .appendMinuteOfHour(2).appendLiteral(":")
                .appendSecondOfMinute(2).appendLiteral(".")
                .appendFractionOfSecond(0, 3)
                .appendTimeZoneOffset("Z", true, 2, 2);
        return new DateTime(ms).toString(fb.toFormatter());
    }

    /**
     * Получить время из милисекунд
     *
     * @param ms милисекунды
     * @return время
     */
    public static String getTimeString(String ms) {
        return getTimeString(Long.parseLong(ms));
    }

    /**
     * получить интервал из милисекунд
     *
     * @param ms милисекунды
     * @return интервал
     */
    public static String getIntervalString(String ms) {
        org.joda.time.Duration d = new org.joda.time.Duration(Long.parseLong(ms));
        return d.toString();
//        long mils = new Long(ms).longValue();
//        long millisecond = mils % 1000;
//        mils = mils / 1000;
//        long second = mils % 60;
//        mils = mils / 60;
//        long minute = mils % 60;
//        mils = mils / 60;
//        long hour = mils % 24;
//        mils = mils / 24;
//        long day = mils % 30;
//        mils = mils / 30;
//        long month = mils % 12;
//        long year = mils / 12;
//
//        String syear = year == 0 ? "" : ""+year+"Y";
//        String smonth = month == 0 ? "" : new Long(month).toString()+"M";
//        String sday = day == 0 ? "" : new Long(day).toString()+"D";
//        String shour = hour == 0 ? "" : new Long(hour).toString()+"H";
//        String sminute = minute == 0 ? "" : new Long(minute).toString()+"M";
//        String ssecond = second == 0 ? "" : new Long(second).toString()+"S";
//        String time = hour != 0 || minute != 0 || second != 0 ? "T" : "";
//        return "P"+syear+smonth+sday+time+shour+sminute+ssecond;
    }

    public static XMLGregorianCalendar newXMLGregorianCalendarInstance(String date) throws DatatypeConfigurationException {
        if (datatype_factory == null) {
            datatype_factory = DatatypeFactory.newInstance();
        }
        return datatype_factory.newXMLGregorianCalendar(date);
    }

    public static XMLGregorianCalendar newXMLGregorianCalendarInstance(Date date) throws DatatypeConfigurationException {
        if (date == null) {
            return null;
        }
        if (gregorian_calendar == null) {
            gregorian_calendar = new GregorianCalendar();
        }
        if (datatype_factory == null) {
            datatype_factory = DatatypeFactory.newInstance();
        }
        gregorian_calendar.setTime(date);
        return datatype_factory.newXMLGregorianCalendar(gregorian_calendar);
    }

    public static GregorianCalendar newGregorianCalendar(Date date) { //throws DatatypeConfigurationException {
        if (date == null) {
            return null;
        }
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTime(date);
        return gc;
    }
}
