package ru.ilb.jfunction.exception;

import java.util.function.Supplier;

/**
 * Utility class for functional extensions
 *
 * @author Peter Kurfer
 */
public final class ExceptionalFunctionUtils {


    private ExceptionalFunctionUtils() {

    }

    /**
     * Functional try-with-resources
     *
     * @param resourceSupplier supplier for the AutoCloseable resource
     * @param consumer         function that consumes the resource and returns a value
     * @param fallbackSupplier supplier for a fallback value
     * @param <T>              Type of the result
     * @param <R>              Type of the AutoCloseable resource
     * @return instance of T supplied by the consumer or fallback value if exception occurred
     */
    public static <T, R extends AutoCloseable> T tryWithResources(
            ExceptionalResourceSupplier<R> resourceSupplier,
            ExceptionalFunction<R, T, Exception> consumer,
            Supplier<T> fallbackSupplier
    ) {
        try (R resource = resourceSupplier.get()) {
            return consumer.apply(resource);
        } catch (Exception e) {
            return fallbackSupplier.get();
        }
    }

    /**
     * Functional try-with-resources with exception translator function
     *
     * @param resourceSupplier supplier for the AutoCloseable resource
     * @param consumer         function that consumes the resource and returns a value
     * @param runtimeExceptionFunction function that converts Exception to RuntimeException
     * @param <T>              Type of the result
     * @param <R>              Type of the AutoCloseable resource
     * @return instance of T supplied by the consumer
     */
    public static <T, R extends AutoCloseable> T tryWithResources(
            ExceptionalResourceSupplier<R> resourceSupplier,
            ExceptionalFunction<R, T, Exception> consumer,
            RuntimeExceptionFunction runtimeExceptionFunction
    ) {
        try (R resource = resourceSupplier.get()) {
            return consumer.apply(resource);
        } catch (Exception e) {
            throw runtimeExceptionFunction.apply(e);
        }
    }

    /**
     * Functional try-with-resources with default exception translator function
     *
     * @param resourceSupplier supplier for the AutoCloseable resource
     * @param consumer         function that consumes the resource and returns a value
     * @param <T>              Type of the result
     * @param <R>              Type of the AutoCloseable resource
     * @return instance of T supplied by the consumer
     */
    public static <T, R extends AutoCloseable> T tryWithResources(
            ExceptionalResourceSupplier<R> resourceSupplier,
            ExceptionalFunction<R, T, Exception> consumer
    ) {
        try (R resource = resourceSupplier.get()) {
            return consumer.apply(resource);
        } catch (Exception e) {
            throw RuntimeExceptionFunction.INSTANCE.apply(e);
        }
    }

}
