/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package ru.ilb.jfunction.marshallers;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;
import javax.inject.Named;

/**
 *
 * @author slavb
 */
@Named
public class MarshallerImpl implements Marshaller {

    private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");
    private static final DateFormat DATETIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    private final Map<Class, MarshallFunction> marshallers = new HashMap<>();
    private final Map<String, MarshallFunction> marshallersString = new HashMap<>();

    public MarshallerImpl() {
        marshallers.put(java.sql.Date.class, d -> DATE_FORMAT.format(d));
        marshallers.put(java.sql.Timestamp.class, d -> DATETIME_FORMAT.format(d));

        // convert to string keys
        marshallers.entrySet().forEach(e -> marshallersString.put(e.getKey().getCanonicalName(), e.getValue()));

    }

    /**
     * Marshalls value to required type
     * Boolean, String and Number values marshalled "as is" to skip quoting
     *
     * @param value
     * @param type
     * @return
     */
    @Override
    public Object marshall(Object value, String type) {
        try {
            Class<?> clazz = type != null ? Class.forName(type) : null;
            return marshall(value, clazz);
        } catch (ClassNotFoundException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * type is marshalled "as is" to skip quoting
     *
     * @param type
     * @return
     */
    protected boolean marshallTypeAsIs(Class type) {
        return Number.class.isAssignableFrom(type) || Boolean.class == type || String.class == type;
    }

    @Override
    public Object marshall(Object value, Class type) {
        if (value == null) {
            return null;
        }
        if (type == null) {
            type = value.getClass();
        }
        if (marshallTypeAsIs(type)) {
            return value;
        }
        MarshallFunction mapper = marshallers.get(type);
        if (mapper != null) {
            return mapper.marshall(value);
        } else {
            return value.toString();
        }
    }

    @Override
    public Object marshall(Object value) {
        return marshall(value, (Class) null);
    }
}
