package ru.mk.pump.web.browsers;

import lombok.NonNull;
import lombok.ToString;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebDriver.Navigation;
import ru.mk.pump.commons.activity.AbstractActivity;
import ru.mk.pump.commons.activity.Activity;
import ru.mk.pump.commons.activity.ActivityListener;
import ru.mk.pump.commons.activity.NamedEvent;
import ru.mk.pump.commons.exception.PumpMessage;
import ru.mk.pump.commons.utils.Strings;
import ru.mk.pump.web.browsers.configuration.BrowserConfig;
import ru.mk.pump.web.exceptions.BrowserException;
import ru.mk.pump.web.common.WebReporter;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Observer;

/**
 * Created by kochetkov-ma on 5/31/17.
 */
@SuppressWarnings("WeakerAccess")
@ToString(of = {"id", "config"}, callSuper = true)
public abstract class AbstractBrowser extends AbstractActivity implements Browser {

    //region FINALS
    private final DriverBuilder builder;

    private final BrowserConfig config;

    private final String id;

    private final WindowManager windows;

    private final DownloadManager downloads;

    private final BrowserActions actions;

    private final LogManager logs;
    //endregion

    private WebDriver driver = null;

    //region INIT
    public AbstractBrowser(@NonNull DriverBuilder builder, @NonNull String uuid) {
        super(uuid);
        this.builder = builder;
        this.config = builder.getConfig();
        this.id = this.config.getType().name() + "_" + uuid;
        this.downloads = new DownloadManager(getConfig());
        this.actions = new BrowserActions(this::getDriver);
        this.logs = new LogManager();
        this.windows = new WindowManager(this);
        addObserver(getDefaultListener());
    }
    //endregion

    @Override
    public String getId() {
        return id;
    }

    @Override
    public Browser start() {
        if (isClosed()) {
            throw new BrowserException("Cannot start browser. Browser has been already closed and this instance of Browser cannot start", this);
        }
        if (!isStarted()) {
            driver = builder.createAndStartDriver();
            if (driver == null) {
                throw new BrowserException("Cannot start browser. Incorrect driver builder", this);
            }
            activate();
        }
        return this;
    }

    @NonNull
    @Override
    public WebDriver getDriver() {
        if (driver == null) {
            throw new BrowserException("Cannot get WebDriver - browser was not started or closed", this);
        }
        return this.driver;
    }

    @Override
    public BrowserConfig getConfig() {
        return builder.getConfig();
    }

    @Override
    public boolean isStarted() {
        return isActive();
    }

    @Override
    public Browser open(String url) {
        getDriver().get(url);
        return this;
    }

    @Override
    public Navigation navigate() {
        return getDriver().navigate();
    }

    @Override
    public Browser refresh() {
        getDriver().navigate().refresh();
        return this;
    }

    @Override
    public WindowManager windows() {
        return windows;
    }

    @Override
    public DownloadManager downloads() {
        return downloads;
    }

    @Override
    public LogManager logs() {
        return logs;
    }

    @Override
    public BrowserActions actions() {
        return actions;
    }

    @Override
    public Activity disable() {
        close();
        return this;
    }

    @Override
    public void close() {
        if (driver != null && !isClosed()) {
            if (isStarted()) {
                forkCloseByBrowserType();
                driver = null;
            } else {
                driver = null;
            }
        }
        super.close();
    }

    @Override
    public Map<String, String> getInfo() {
        final LinkedHashMap<String, String> result = new LinkedHashMap<>();
        result.put("id", id);
        result.put("builder", builder.getClass().getSimpleName());
        result.put("config", config.toString());
        return result;
    }

    protected void forkCloseByBrowserType() {
        /*for special closing logic some browser types*/
        driver.quit();
    }

    protected Observer getDefaultListener() {
        return new ActivityListener() {
            @Override
            public void onClose(NamedEvent namedEvent, Activity activity) {
                report(namedEvent, activity);
            }

            @Override
            public void onActivate(NamedEvent namedEvent, Activity activity) {
                report(namedEvent, activity);
            }

            @Override
            public void onDisable(NamedEvent namedEvent, Activity activity) {
                report(namedEvent, activity);
            }

            private void report(NamedEvent namedEvent, Activity activity) {
                final PumpMessage msg = new PumpMessage(Strings.toString(namedEvent))
                        .addExtraInfo(((AbstractBrowser) activity));
                WebReporter.getReporter()
                        .info("Browser has been " + namedEvent.getName(), msg.toPrettyString(),
                                WebReporter.getReporter().attachments().dummy());
            }
        };
    }
}
