package ru.ivk1800.riflesso

import org.jetbrains.kotlin.backend.common.extensions.IrPluginContext
import org.jetbrains.kotlin.ir.SourceRangeInfo
import org.jetbrains.kotlin.ir.UNDEFINED_OFFSET
import org.jetbrains.kotlin.ir.declarations.IrEnumEntry
import org.jetbrains.kotlin.ir.declarations.IrValueDeclaration
import org.jetbrains.kotlin.ir.declarations.IrValueParameter
import org.jetbrains.kotlin.ir.declarations.IrVariable
import org.jetbrains.kotlin.ir.expressions.IrExpression
import org.jetbrains.kotlin.ir.expressions.impl.IrCallImpl
import org.jetbrains.kotlin.ir.expressions.impl.IrConstImpl
import org.jetbrains.kotlin.ir.expressions.impl.IrConstructorCallImpl
import org.jetbrains.kotlin.ir.expressions.impl.IrGetEnumValueImpl
import org.jetbrains.kotlin.ir.expressions.impl.IrGetValueImpl
import org.jetbrains.kotlin.ir.expressions.impl.IrVarargImpl
import org.jetbrains.kotlin.ir.types.classFqName
import org.jetbrains.kotlin.ir.types.typeWith
import org.jetbrains.kotlin.ir.util.constructors
import org.jetbrains.kotlin.ir.util.defaultType
import org.jetbrains.kotlin.ir.util.primaryConstructor
import org.jetbrains.kotlin.name.CallableId
import org.jetbrains.kotlin.name.ClassId
import org.jetbrains.kotlin.name.FqName
import org.jetbrains.kotlin.name.Name
import java.util.UUID

class HighlightCallArgumentsBuilder(
    private val context: IrPluginContext,
) {
    private val listOfSymbol by lazy {
        context.referenceFunctions(CallableId(FqName("kotlin.collections"), Name.identifier("listOf")))
            .toList()[1]
    }

    private val paramStateClassId = ClassId.topLevel(FqName("ru.ivk1800.riflesso.ParamState"))

    private val paramStateSymbol by lazy {
        context.referenceClass(paramStateClassId)
            ?: error("ParamState not found on compilation classpath: $paramStateClassId")
    }

    private val valueDeclarationClassSymbol by lazy {
        context.referenceClass(ClassId.topLevel(FqName("ru.ivk1800.riflesso.ValueDeclaration")))
            ?: error("ValueDeclaration class not found")
    }

    private val valueDeclarationClassCtor by lazy {
        val paramClass = valueDeclarationClassSymbol.owner
        paramClass.primaryConstructor ?: paramClass.constructors.firstOrNull()
        ?: error("Param constructor not found")
    }

    private val highlightTypeClassSymbol by lazy {
        context.referenceClass(ClassId.topLevel(FqName("ru.ivk1800.riflesso.HighlightType")))
            ?: error("HighlightType not found")
    }

    private val recompositionEnumEntry by lazy {
        val name = Name.identifier("Recomposition")
        val stateClass = highlightTypeClassSymbol.owner
        stateClass.declarations.filterIsInstance<IrEnumEntry>()
            .firstOrNull { it.name == name }
            ?: error("HighlightType $name not found")
    }

    private val skipEnumEntry by lazy {
        val name = Name.identifier("Skip")
        val stateClass = highlightTypeClassSymbol.owner
        stateClass.declarations.filterIsInstance<IrEnumEntry>()
            .firstOrNull { it.name == name }
            ?: error("HighlightType $name not found")
    }

    fun build(
        callableId: String,
        fileName: String,
        callablePackageName: String?,
        callableName: String?,
        parentFunctionName: String?,
        parentFunctionPackageName: String?,
        bodySourceRangeInfo: SourceRangeInfo,
        parameters: List<IrValueParameter>,
        variables: List<IrVariable>,
        type: HighlightType,
    ): List<IrExpression> =
        listOf(
            // filePath
            IrConstImpl.string(
                startOffset = UNDEFINED_OFFSET,
                endOffset = UNDEFINED_OFFSET,
                type = context.irBuiltIns.stringType,
                value = fileName,
            ),
            // fileId
            IrConstImpl.string(
                startOffset = UNDEFINED_OFFSET,
                endOffset = UNDEFINED_OFFSET,
                type = context.irBuiltIns.stringType,
                value = UUID.nameUUIDFromBytes(fileName.toByteArray()).toString(),
            ),
            // startOffset
            IrConstImpl.int(
                startOffset = UNDEFINED_OFFSET,
                endOffset = UNDEFINED_OFFSET,
                type = context.irBuiltIns.intType,
                value = bodySourceRangeInfo.startOffset,
            ),
            // endOffset
            IrConstImpl.int(
                startOffset = UNDEFINED_OFFSET,
                endOffset = UNDEFINED_OFFSET,
                type = context.irBuiltIns.intType,
                value = bodySourceRangeInfo.endOffset,
            ),
            // callableName
            if (callableName == null) {
                IrConstImpl.constNull(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                )
            } else {
                IrConstImpl.string(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                    value = callableName,
                )
            },
            // callableId
            IrConstImpl.string(
                startOffset = UNDEFINED_OFFSET,
                endOffset = UNDEFINED_OFFSET,
                type = context.irBuiltIns.stringType,
                value = callableId,
            ),
            // parentFunctionName
            if (parentFunctionName == null) {
                IrConstImpl.constNull(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                )
            } else {
                IrConstImpl.string(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                    value = parentFunctionName,
                )
            },
            // callablePackageName
            if (callablePackageName == null) {
                IrConstImpl.constNull(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                )
            } else {
                IrConstImpl.string(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                    value = callablePackageName,
                )
            },
            // parentFunctionPackageName
            if (parentFunctionPackageName == null) {
                IrConstImpl.constNull(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                )
            } else {
                IrConstImpl.string(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                    value = parentFunctionPackageName,
                )
            },
            // params
            createValueDeclarationsList(parameters),
            // variables
            createValueDeclarationsList(variables),
            // type
            createType(type),
        )

    private fun createType(
        type: HighlightType,
    ): IrGetEnumValueImpl {
        val stateClass = highlightTypeClassSymbol.owner

        val symbol = when (type) {
            HighlightType.Recomposition -> recompositionEnumEntry
            HighlightType.Skip -> skipEnumEntry
        }.symbol

        return IrGetEnumValueImpl(
            startOffset = UNDEFINED_OFFSET,
            endOffset = UNDEFINED_OFFSET,
            type = stateClass.defaultType,
            symbol = symbol,
        )
    }

    private fun createValueDeclarationsList(values: List<IrValueDeclaration>): IrCallImpl {
        val vararg = IrVarargImpl(
            startOffset = UNDEFINED_OFFSET,
            endOffset = UNDEFINED_OFFSET,
            type = context.irBuiltIns.arrayClass.typeWith(valueDeclarationClassSymbol.owner.defaultType),
            varargElementType = valueDeclarationClassSymbol.owner.defaultType,
            elements = values.map { variable ->
                val ctor = valueDeclarationClassCtor
                val ctorCall = IrConstructorCallImpl(
                    startOffset = 0,
                    endOffset = 0,
                    type = valueDeclarationClassSymbol.owner.defaultType,
                    symbol = ctor.symbol,
                    typeArgumentsCount = ctor.typeParameters.size,
                    constructorTypeArgumentsCount = ctor.typeParameters.size,
                    valueArgumentsCount = 3,
                )

                // name
                val name = IrConstImpl.string(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                    value = variable.name.asString(),
                )

                // type
                val type = IrConstImpl.string(
                    startOffset = UNDEFINED_OFFSET,
                    endOffset = UNDEFINED_OFFSET,
                    type = context.irBuiltIns.stringType,
                    value = variable.type.classFqName?.shortName()?.asString().orEmpty(),
                )

                // value
                val value = IrGetValueImpl(UNDEFINED_OFFSET, UNDEFINED_OFFSET, variable.symbol)

                ctorCall.putValueArgument(0, name)
                ctorCall.putValueArgument(1, type)
                ctorCall.putValueArgument(2, value)
                ctorCall
            },
        )


        val listCall = IrCallImpl.fromSymbolOwner(
            startOffset = UNDEFINED_OFFSET,
            endOffset = UNDEFINED_OFFSET,
            type = context.irBuiltIns.listClass.typeWith(paramStateSymbol.owner.defaultType),
            symbol = listOfSymbol,
        )
        listCall.putTypeArgument(0, paramStateSymbol.owner.defaultType)
        listCall.putValueArgument(0, vararg)
        return listCall
    }
}
