package ru.ivk1800.riflesso

import org.jetbrains.kotlin.backend.common.extensions.IrPluginContext
import org.jetbrains.kotlin.ir.IrElement
import org.jetbrains.kotlin.ir.IrFileEntry
import org.jetbrains.kotlin.ir.IrStatement
import org.jetbrains.kotlin.ir.UNDEFINED_OFFSET
import org.jetbrains.kotlin.ir.declarations.IrFile
import org.jetbrains.kotlin.ir.declarations.IrSimpleFunction
import org.jetbrains.kotlin.ir.declarations.IrValueParameter
import org.jetbrains.kotlin.ir.declarations.IrVariable
import org.jetbrains.kotlin.ir.expressions.IrBlock
import org.jetbrains.kotlin.ir.expressions.IrBlockBody
import org.jetbrains.kotlin.ir.expressions.IrCall
import org.jetbrains.kotlin.ir.expressions.IrFunctionExpression
import org.jetbrains.kotlin.ir.expressions.impl.IrBlockImpl
import org.jetbrains.kotlin.ir.expressions.impl.IrCallImpl
import org.jetbrains.kotlin.ir.symbols.IrSimpleFunctionSymbol
import org.jetbrains.kotlin.ir.util.fqNameWhenAvailable
import org.jetbrains.kotlin.ir.util.hasAnnotation
import org.jetbrains.kotlin.ir.visitors.IrElementVisitor
import org.jetbrains.kotlin.name.CallableId
import org.jetbrains.kotlin.name.FqName
import org.jetbrains.kotlin.name.Name
import java.util.UUID

class HighlightInjector(
    private val context: IrPluginContext,
) : IrElementVisitor<Unit, HighlightInjector.Data> {
    private val highlightCallArgumentsBuilder = HighlightCallArgumentsBuilder(context)

    private val composableAnnotation = FqName("androidx.compose.runtime.Composable")

    private val highlightSymbol: IrSimpleFunctionSymbol =
        context.referenceFunctions(CallableId(FqName("ru.ivk1800.riflesso"), Name.identifier("highlight"))).first()
    private val endDefaultsName = Name.identifier("endDefaults")

    class Data(var irFile: IrFile? = null)

    override fun visitElement(element: IrElement, data: Data) {

        element.acceptChildren(this, data)
    }

    private val irFileStack = ArrayDeque<IrFile>()

    override fun visitFile(declaration: IrFile, data: Data) {
        data.irFile = declaration
        irFileStack.addLast(declaration)
        declaration.acceptChildren(this, data)
        irFileStack.removeLast()
    }

    private val simpleFunctionStack = ArrayDeque<IrSimpleFunction>()

    override fun visitSimpleFunction(declaration: IrSimpleFunction, data: Data) {
        if (declaration.annotations.hasAnnotation(composableAnnotation)) {
            val irFile = data.irFile ?: return
            injectHighlightCall(irFile, declaration)

            simpleFunctionStack.addLast(declaration)
            declaration.acceptChildren(this, data)
            simpleFunctionStack.removeLast()
        }
    }

    override fun visitFunctionExpression(expression: IrFunctionExpression, data: Data) {
        val declaration: IrSimpleFunction = expression.function

        if (declaration.annotations.hasAnnotation(composableAnnotation)) {
            val irFile = data.irFile ?: return
            injectHighlightCall(irFile, declaration)
            expression.function.acceptChildren(this, data)
        }
    }

    private fun injectHighlightCall(irFile: IrFile, declaration: IrSimpleFunction) {
        val body: IrBlockBody = declaration.body as? IrBlockBody ?: return

        val composableSkippingIrWhen = ComposableSkippingFinder.find(declaration)

        if (composableSkippingIrWhen != null) {
            val callableId = UUID.randomUUID().toString()

            val callableFqName: FqName? = declaration.fqNameWhenAvailable
            val parentFunctionFqName = simpleFunctionStack.lastOrNull()?.fqNameWhenAvailable
            val skipToGroupEnd = composableSkippingIrWhen.branches[1].result as IrCall
            val statements = mutableListOf<IrStatement>(skipToGroupEnd)
            val callableName = callableFqName?.shortName()?.asString() ?: declaration.name.asString()
            val callablePackageName = callableFqName?.parent()?.asString()
            val parentFunctionName = parentFunctionFqName?.shortName()?.asString()
            val parentFunctionPackageName = parentFunctionFqName?.parent()?.asString()
            val parameters = declaration.valueParameters
            val variables = body.statements.getDirties()
            val fileEntry = irFile.fileEntry

            // region skip

            injectHighlightCall(
                callableId = callableId,
                fileEntry = fileEntry,
                body = body,
                irFile = irFile,
                callableName = callableName,
                callablePackageName = callablePackageName,
                parentFunctionName = parentFunctionName,
                parentFunctionPackageName = parentFunctionPackageName,
                parameters = parameters,
                variables = variables,
                type = HighlightType.Skip,
                indexToInject = 0,
                statementsToInject = statements,
            )

            composableSkippingIrWhen.branches[1].result = IrBlockImpl(
                startOffset = UNDEFINED_OFFSET,
                endOffset = UNDEFINED_OFFSET,
                type = context.irBuiltIns.unitType,
                origin = null,
                statements = statements,
            )

            // endregion

            // region recomposition

            val blockToInject = composableSkippingIrWhen.branches[0].result as IrBlock

            injectHighlightCall(
                callableId = callableId,
                fileEntry = fileEntry,
                body = body,
                irFile = irFile,
                callableName = callableName,
                callablePackageName = callablePackageName,
                parentFunctionName = parentFunctionName,
                parentFunctionPackageName = parentFunctionPackageName,
                parameters = parameters,
                variables = variables,
                type = HighlightType.Recomposition,
                indexToInject = blockToInject.statements.findIndexToInject(),
                statementsToInject = blockToInject.statements,
            )

            // endregion
        }
    }

    private fun List<IrStatement>.getDirties(): List<IrVariable> =
        buildList {
            this@getDirties.forEach { statement ->
                if (statement is IrVariable && statement.name.asString().startsWith("\$dirty")) {
                    add(statement)
                }
            }
        }

    /**
     *            $composer2.startDefaults();
     *             ComposerKt.sourceInformation($composer2, "165@5964L23");
     *             if (($changed & 1) != 0 && !$composer2.getDefaultsInvalid()) {
     *                 $composer2.skipToGroupEnd();
     *                 if ((i & 1) != 0) {
     *                     $dirty &= -15;
     *                 }
     *             } else if ((i & 1) != 0) {
     *                 state = LazyListStateKt.rememberLazyListState(0, 0, $composer2, 0, 3);
     *                 $dirty &= -15;
     *             }
     *             $composer2.endDefaults();
     *
     * @Composable
     * private fun MyList(
     *     state: LazyListState = rememberLazyListState()
     * ) {
     *     LazyColumn(state = state) {  }
     * }
     */
    private fun List<IrStatement>.findIndexToInject(): Int =
        indexOfFirst { it is IrCall && it.symbol.owner.name == endDefaultsName }
            .takeIf { it >= 0 }
            ?.let { it + 1 } ?: 0

    private fun injectHighlightCall(
        callableId: String,
        fileEntry: IrFileEntry,
        body: IrBlockBody,
        indexToInject: Int,
        statementsToInject: MutableList<IrStatement>,
        irFile: IrFile,
        callablePackageName: String?,
        callableName: String?,
        parentFunctionName: String?,
        parentFunctionPackageName: String?,
        parameters: List<IrValueParameter>,
        variables: List<IrVariable>,
        type: HighlightType,
    ) {
        val bodySourceRangeInfo = fileEntry.getSourceRangeInfo(
            beginOffset = body.startOffset,
            endOffset = body.endOffset,
        )

        val fileName = irFile.fileEntry.name

        val arguments = highlightCallArgumentsBuilder.build(
            fileName = fileName,
            callablePackageName = callablePackageName,
            callableName = callableName,
            parentFunctionName = parentFunctionName,
            parentFunctionPackageName = parentFunctionPackageName,
            bodySourceRangeInfo = bodySourceRangeInfo,
            parameters = parameters,
            variables = variables,
            type = type,
            callableId = callableId,
        )

        val highlightCall = IrCallImpl.fromSymbolOwner(
            startOffset = UNDEFINED_OFFSET,
            endOffset = UNDEFINED_OFFSET,
            type = context.irBuiltIns.unitType,
            symbol = highlightSymbol,
        )

        arguments.forEachIndexed { index, arg -> highlightCall.putValueArgument(index, arg) }
        statementsToInject.add(indexToInject, highlightCall)
    }
}
