/*
 * Decompiled with CFR 0.152.
 */
package one.nio.server;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.LongAdder;
import one.nio.mgt.Management;
import one.nio.net.Selector;
import one.nio.net.Session;
import one.nio.net.Socket;
import one.nio.server.AcceptorConfig;
import one.nio.server.AcceptorThread;
import one.nio.server.CleanupThread;
import one.nio.server.RejectedSessionException;
import one.nio.server.SelectorThread;
import one.nio.server.ServerConfig;
import one.nio.server.ServerMXBean;
import one.nio.server.WorkerPool;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class Server
implements ServerMXBean {
    private static final Log log = LogFactory.getLog(Server.class);
    private final LongAdder requestsProcessed = new LongAdder();
    private final LongAdder requestsRejected = new LongAdder();
    private volatile SelectorStats selectorStats;
    private volatile QueueStats queueStats;
    protected final int port;
    protected final CountDownLatch startSync;
    protected volatile AcceptorThread[] acceptors;
    protected volatile SelectorThread[] selectors;
    protected boolean useWorkers;
    protected final WorkerPool workers;
    protected final CleanupThread cleanup;

    public Server(ServerConfig config) throws IOException {
        ArrayList<AcceptorThread> acceptors = new ArrayList<AcceptorThread>();
        for (AcceptorConfig ac : config.acceptors) {
            for (int i = 0; i < ac.threads; ++i) {
                acceptors.add(new AcceptorThread(this, ac, i));
            }
        }
        if (acceptors.isEmpty()) {
            throw new IllegalArgumentException("No configured acceptors");
        }
        this.acceptors = acceptors.toArray(new AcceptorThread[0]);
        this.startSync = new CountDownLatch(this.acceptors.length);
        this.port = this.acceptors[0].port;
        int processors = Runtime.getRuntime().availableProcessors();
        SelectorThread[] selectors = new SelectorThread[config.selectors != 0 ? config.selectors : processors];
        for (int i = 0; i < selectors.length; ++i) {
            selectors[i] = new SelectorThread(i, config.affinity ? i % processors : -1, config.schedulingPolicy);
            selectors[i].setPriority(config.threadPriority);
        }
        this.selectors = selectors;
        this.useWorkers = config.maxWorkers > 0;
        this.workers = new WorkerPool(config.minWorkers, this.useWorkers ? config.maxWorkers : 2, (long)config.queueTime, config.threadPriority, config.schedulingPolicy);
        this.cleanup = new CleanupThread(selectors, config.keepAlive);
        this.selectorStats = new SelectorStats();
        this.queueStats = new QueueStats();
    }

    public synchronized void reconfigure(ServerConfig config) throws IOException {
        boolean bl = this.useWorkers = config.maxWorkers > 0;
        if (config.minWorkers > this.workers.getMaximumPoolSize()) {
            this.workers.setMaximumPoolSize(this.useWorkers ? config.maxWorkers : 2);
            this.workers.setCorePoolSize(config.minWorkers);
        } else {
            this.workers.setCorePoolSize(config.minWorkers);
            this.workers.setMaximumPoolSize(this.useWorkers ? config.maxWorkers : 2);
        }
        this.workers.setQueueTime(config.queueTime);
        AcceptorThread[] oldAcceptors = (AcceptorThread[])this.acceptors.clone();
        ArrayList<AcceptorThread> newAcceptors = new ArrayList<AcceptorThread>();
        for (AcceptorConfig ac : config.acceptors) {
            int threads = 0;
            for (int i = 0; i < oldAcceptors.length; ++i) {
                AcceptorThread oldAcceptor = oldAcceptors[i];
                if (oldAcceptor == null || oldAcceptor.port != ac.port || !oldAcceptor.address.equals(ac.address) || ++threads > ac.threads) continue;
                log.info((Object)("Reconfiguring acceptor: " + oldAcceptor.getName()));
                oldAcceptor.reconfigure(ac);
                oldAcceptors[i] = null;
                newAcceptors.add(oldAcceptor);
            }
            while (threads < ac.threads) {
                AcceptorThread newAcceptor = new AcceptorThread(this, ac, threads);
                log.info((Object)("New acceptor: " + newAcceptor.getName()));
                newAcceptor.start();
                newAcceptors.add(newAcceptor);
                ++threads;
            }
        }
        for (AcceptorThread oldAcceptor : oldAcceptors) {
            if (oldAcceptor == null) continue;
            log.info((Object)("Stopping acceptor: " + oldAcceptor.getName()));
            oldAcceptor.shutdown();
        }
        this.acceptors = newAcceptors.toArray(new AcceptorThread[0]);
        int processors = Runtime.getRuntime().availableProcessors();
        SelectorThread[] selectors = this.selectors;
        if (config.selectors > selectors.length) {
            SelectorThread[] newSelectors = Arrays.copyOf(selectors, config.selectors);
            for (int i = selectors.length; i < config.selectors; ++i) {
                newSelectors[i] = new SelectorThread(i, config.affinity ? i % processors : -1, config.schedulingPolicy);
                newSelectors[i].setPriority(config.threadPriority);
                newSelectors[i].start();
            }
            this.selectors = newSelectors;
        }
        this.cleanup.update(this.selectors, config.keepAlive);
    }

    public synchronized void start() {
        for (SelectorThread selectorThread : this.selectors) {
            selectorThread.start();
        }
        for (Thread thread : this.acceptors) {
            thread.start();
        }
        try {
            this.startSync.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.cleanup.start();
        Management.registerMXBean(this, "one.nio.server:type=Server,port=" + this.port);
    }

    public synchronized void stop() {
        Management.unregisterMXBean("one.nio.server:type=Server,port=" + this.port);
        this.cleanup.shutdown();
        for (AcceptorThread acceptorThread : this.acceptors) {
            acceptorThread.shutdown();
        }
        for (Thread thread : this.selectors) {
            ((SelectorThread)thread).shutdown();
        }
        this.workers.gracefulShutdown();
    }

    public void registerShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread("Server Shutdown"){

            @Override
            public void run() {
                log.info((Object)"Server is shutting down...");
                Server.this.stop();
                log.info((Object)"Server stopped");
            }
        });
    }

    protected Session createSession(Socket socket) throws RejectedSessionException {
        return new Session(socket);
    }

    protected void register(Session session) {
        this.getSmallestSelector().register(session);
    }

    private Selector getSmallestSelector() {
        SelectorThread[] selectors = this.selectors;
        ThreadLocalRandom r = ThreadLocalRandom.current();
        Selector a = selectors[r.nextInt((int)selectors.length)].selector;
        Selector b = selectors[r.nextInt((int)selectors.length)].selector;
        return a.size() < b.size() ? a : b;
    }

    public final void incRequestsProcessed() {
        this.requestsProcessed.increment();
    }

    public final void incRequestsRejected() {
        this.requestsRejected.increment();
    }

    @Override
    public int getConnections() {
        int result = 0;
        for (SelectorThread selector : this.selectors) {
            result += selector.selector.size();
        }
        return result;
    }

    @Override
    public long getKeepAlive() {
        return this.cleanup.getKeepAlive();
    }

    @Override
    public boolean getWorkersUsed() {
        return this.useWorkers;
    }

    @Override
    public int getWorkers() {
        return this.workers.getPoolSize();
    }

    @Override
    public int getWorkersActive() {
        return this.workers.getActiveCount();
    }

    @Override
    public long getAcceptedSessions() {
        long result = 0L;
        for (AcceptorThread acceptor : this.acceptors) {
            result += acceptor.acceptedSessions;
        }
        return result;
    }

    @Override
    public long getRejectedSessions() {
        long result = 0L;
        for (AcceptorThread acceptor : this.acceptors) {
            result += acceptor.rejectedSessions;
        }
        return result;
    }

    @Override
    public int getSelectorCount() {
        return this.selectors.length;
    }

    @Override
    public double getSelectorAvgReady() {
        SelectorStats selectorStats = this.getSelectorStats();
        return selectorStats.operations == 0L ? 0.0 : (double)selectorStats.sessions / (double)selectorStats.operations;
    }

    @Override
    public int getSelectorMaxReady() {
        return this.getSelectorStats().maxReady;
    }

    @Override
    public long getSelectorOperations() {
        return this.getSelectorStats().operations;
    }

    @Override
    public long getSelectorSessions() {
        return this.getSelectorStats().sessions;
    }

    @Override
    public double getQueueAvgLength() {
        QueueStats queueStats = this.getQueueStats();
        return queueStats.sessions == 0 ? 0.0 : (double)queueStats.totalLength / (double)queueStats.sessions;
    }

    @Override
    public long getQueueAvgBytes() {
        QueueStats queueStats = this.getQueueStats();
        return queueStats.sessions == 0 ? 0L : queueStats.totalBytes / (long)queueStats.sessions;
    }

    @Override
    public long getQueueMaxLength() {
        return this.getQueueStats().maxLength;
    }

    @Override
    public long getQueueMaxBytes() {
        return this.getQueueStats().maxBytes;
    }

    @Override
    public long getRequestsProcessed() {
        return this.requestsProcessed.sum();
    }

    @Override
    public long getRequestsRejected() {
        return this.requestsRejected.sum();
    }

    @Override
    public synchronized void reset() {
        for (AcceptorThread acceptorThread : this.acceptors) {
            acceptorThread.acceptedSessions = 0L;
            acceptorThread.rejectedSessions = 0L;
        }
        for (Thread thread : this.selectors) {
            ((SelectorThread)thread).operations = 0L;
            ((SelectorThread)thread).sessions = 0L;
            ((SelectorThread)thread).maxReady = 0;
        }
        this.requestsProcessed.reset();
        this.requestsRejected.reset();
    }

    public final void asyncExecute(Runnable command) {
        this.workers.execute(command);
    }

    private synchronized SelectorStats getSelectorStats() {
        SelectorStats selectorStats = this.selectorStats;
        long currentTime = System.currentTimeMillis();
        if (currentTime < selectorStats.expireTime) {
            return selectorStats;
        }
        selectorStats = new SelectorStats();
        selectorStats.expireTime = currentTime + 1000L;
        for (SelectorThread selector : this.selectors) {
            selectorStats.operations += selector.operations;
            selectorStats.sessions += selector.sessions;
            selectorStats.maxReady = Math.max(selectorStats.maxReady, selector.maxReady);
        }
        this.selectorStats = selectorStats;
        return selectorStats;
    }

    private synchronized QueueStats getQueueStats() {
        QueueStats queueStats = this.queueStats;
        long currentTime = System.currentTimeMillis();
        if (currentTime < queueStats.expireTime) {
            return queueStats;
        }
        queueStats = new QueueStats();
        queueStats.expireTime = currentTime + 1000L;
        long[] stats = new long[2];
        for (SelectorThread selector : this.selectors) {
            for (Session session : selector.selector) {
                session.getQueueStats(stats);
                ++queueStats.sessions;
                queueStats.totalLength += stats[0];
                queueStats.totalBytes += stats[1];
                if (stats[0] > queueStats.maxLength) {
                    queueStats.maxLength = stats[0];
                }
                if (stats[1] <= queueStats.maxBytes) continue;
                queueStats.maxBytes = stats[1];
            }
        }
        this.queueStats = queueStats;
        return queueStats;
    }

    private static final class QueueStats {
        long expireTime;
        long totalLength;
        long totalBytes;
        long maxLength;
        long maxBytes;
        int sessions;

        private QueueStats() {
        }
    }

    private static final class SelectorStats {
        long expireTime;
        long operations;
        long sessions;
        int maxReady;

        private SelectorStats() {
        }
    }
}

