/*
 * Decompiled with CFR 0.152.
 */
package one.nio.server;

import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.LongAdder;
import one.nio.mgt.Management;
import one.nio.net.Selector;
import one.nio.net.Session;
import one.nio.net.Socket;
import one.nio.server.CleanupThread;
import one.nio.server.RejectedSessionException;
import one.nio.server.SelectorThread;
import one.nio.server.ServerConfig;
import one.nio.server.ServerMXBean;
import one.nio.server.WorkerPool;
import one.nio.server.acceptor.Acceptor;
import one.nio.server.acceptor.AcceptorFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Server
implements ServerMXBean {
    private static final Logger log = LoggerFactory.getLogger(Server.class);
    private final LongAdder requestsProcessed = new LongAdder();
    private final LongAdder requestsRejected = new LongAdder();
    private volatile SelectorStats selectorStats;
    private volatile QueueStats queueStats;
    protected final int port;
    protected volatile Acceptor acceptor;
    protected volatile SelectorThread[] selectors;
    protected boolean useWorkers;
    protected final WorkerPool workers;
    protected final CleanupThread cleanup;
    protected boolean closeSessions;
    protected boolean pinAcceptors;

    public Server(ServerConfig config) throws IOException {
        this.acceptor = AcceptorFactory.get(config).create(this, config.acceptors);
        this.port = this.acceptor.getSinglePort();
        int processors = Runtime.getRuntime().availableProcessors();
        SelectorThread[] selectors = new SelectorThread[config.selectors != 0 ? config.selectors : processors];
        for (int i = 0; i < selectors.length; ++i) {
            String threadName = config.formatSelectorThreadName(i);
            selectors[i] = new SelectorThread(i, config.affinity ? i % processors : -1, config.schedulingPolicy, threadName);
            selectors[i].setPriority(config.threadPriority);
        }
        this.selectors = selectors;
        this.useWorkers = config.maxWorkers > 0;
        this.workers = new WorkerPool(config.minWorkers, this.useWorkers ? config.maxWorkers : 2, (long)config.queueTime, config.threadPriority, config.schedulingPolicy);
        this.cleanup = new CleanupThread(selectors, config.keepAlive);
        this.closeSessions = config.closeSessions;
        this.pinAcceptors = config.pinAcceptors;
        this.selectorStats = new SelectorStats();
        this.queueStats = new QueueStats();
    }

    public synchronized void reconfigure(ServerConfig config) throws IOException {
        boolean bl = this.useWorkers = config.maxWorkers > 0;
        if (config.minWorkers > this.workers.getMaximumPoolSize()) {
            this.workers.setMaximumPoolSize(this.useWorkers ? config.maxWorkers : 2);
            this.workers.setCorePoolSize(config.minWorkers);
        } else {
            this.workers.setCorePoolSize(config.minWorkers);
            this.workers.setMaximumPoolSize(this.useWorkers ? config.maxWorkers : 2);
        }
        this.workers.setQueueTime(config.queueTime);
        this.acceptor.reconfigure(config.acceptors);
        int processors = Runtime.getRuntime().availableProcessors();
        SelectorThread[] selectors = this.selectors;
        if (config.selectors > selectors.length) {
            SelectorThread[] newSelectors = Arrays.copyOf(selectors, config.selectors);
            for (int i = selectors.length; i < config.selectors; ++i) {
                String threadName = config.formatSelectorThreadName(i);
                newSelectors[i] = new SelectorThread(i, config.affinity ? i % processors : -1, config.schedulingPolicy, threadName);
                newSelectors[i].setPriority(config.threadPriority);
                newSelectors[i].start();
            }
            this.selectors = newSelectors;
        }
        this.cleanup.update(this.selectors, config.keepAlive);
        this.closeSessions = config.closeSessions;
        this.pinAcceptors = config.pinAcceptors;
    }

    public synchronized void start() {
        for (SelectorThread selector : this.selectors) {
            selector.start();
        }
        this.acceptor.start();
        try {
            this.acceptor.syncStart();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        this.cleanup.start();
        Management.registerMXBean(this, "one.nio.server:type=Server,port=" + this.port);
    }

    public synchronized void stop() {
        Management.unregisterMXBean("one.nio.server:type=Server,port=" + this.port);
        this.cleanup.shutdown();
        this.acceptor.shutdown();
        for (SelectorThread selector : this.selectors) {
            if (this.closeSessions) {
                for (Session session : selector.selector) {
                    session.close();
                }
            }
            selector.shutdown();
        }
        this.workers.gracefulShutdown();
    }

    public void registerShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread("Server Shutdown"){

            @Override
            public void run() {
                log.info("Server is shutting down...");
                Server.this.stop();
                log.info("Server stopped");
            }
        });
    }

    public Session createSession(Socket socket) throws RejectedSessionException {
        return new Session(socket);
    }

    public void register(Session session, int acceptorIndex, int acceptorGroupSize) {
        if (this.pinAcceptors) {
            this.getSmallestPinnedSelector(acceptorIndex, acceptorGroupSize).register(session);
            return;
        }
        this.register(session);
    }

    public void register(Session session) {
        this.getSmallestSelector().register(session);
    }

    private Selector getSmallestPinnedSelector(int idx, int total) {
        Selector chosen;
        SelectorThread[] selectors = this.selectors;
        if (total >= selectors.length) {
            chosen = selectors[idx % selectors.length].selector;
        } else {
            int q = selectors.length / total;
            if (q * total + idx < selectors.length) {
                ++q;
            }
            if (q == 1) {
                chosen = selectors[idx].selector;
            } else {
                ThreadLocalRandom r = ThreadLocalRandom.current();
                Selector a = selectors[r.nextInt((int)q) * total + idx].selector;
                Selector b = selectors[r.nextInt((int)q) * total + idx].selector;
                chosen = a.size() < b.size() ? a : b;
            }
        }
        return chosen;
    }

    private Selector getSmallestSelector() {
        SelectorThread[] selectors = this.selectors;
        ThreadLocalRandom r = ThreadLocalRandom.current();
        Selector a = selectors[r.nextInt((int)selectors.length)].selector;
        Selector b = selectors[r.nextInt((int)selectors.length)].selector;
        return a.size() < b.size() ? a : b;
    }

    public final void incRequestsProcessed() {
        this.requestsProcessed.increment();
    }

    public final void incRequestsRejected() {
        this.requestsRejected.increment();
    }

    @Override
    public int getConnections() {
        int result = 0;
        for (SelectorThread selector : this.selectors) {
            result += selector.selector.size();
        }
        return result;
    }

    @Override
    public long getKeepAlive() {
        return this.cleanup.getKeepAlive();
    }

    @Override
    public boolean getWorkersUsed() {
        return this.useWorkers;
    }

    @Override
    public int getWorkers() {
        return this.workers.getPoolSize();
    }

    @Override
    public int getWorkersActive() {
        return this.workers.getActiveCount();
    }

    @Override
    public long getAcceptedSessions() {
        return this.acceptor.getAcceptedSessions();
    }

    @Override
    public long getRejectedSessions() {
        return this.acceptor.getRejectedSessions();
    }

    @Override
    public int getSelectorCount() {
        return this.selectors.length;
    }

    @Override
    public double getSelectorAvgReady() {
        SelectorStats selectorStats = this.getSelectorStats();
        return selectorStats.operations == 0L ? 0.0 : (double)selectorStats.sessions / (double)selectorStats.operations;
    }

    @Override
    public int getSelectorMaxReady() {
        return this.getSelectorStats().maxReady;
    }

    @Override
    public long getSelectorOperations() {
        return this.getSelectorStats().operations;
    }

    @Override
    public long getSelectorSessions() {
        return this.getSelectorStats().sessions;
    }

    @Override
    public double getQueueAvgLength() {
        QueueStats queueStats = this.getQueueStats();
        return queueStats.sessions == 0 ? 0.0 : (double)queueStats.totalLength / (double)queueStats.sessions;
    }

    @Override
    public long getQueueAvgBytes() {
        QueueStats queueStats = this.getQueueStats();
        return queueStats.sessions == 0 ? 0L : queueStats.totalBytes / (long)queueStats.sessions;
    }

    @Override
    public long getQueueMaxLength() {
        return this.getQueueStats().maxLength;
    }

    @Override
    public long getQueueMaxBytes() {
        return this.getQueueStats().maxBytes;
    }

    @Override
    public long getRequestsProcessed() {
        return this.requestsProcessed.sum();
    }

    @Override
    public long getRequestsRejected() {
        return this.requestsRejected.sum();
    }

    @Override
    public synchronized void reset() {
        this.acceptor.resetCounters();
        for (SelectorThread selector : this.selectors) {
            selector.operations = 0L;
            selector.sessions = 0L;
            selector.maxReady = 0;
        }
        this.requestsProcessed.reset();
        this.requestsRejected.reset();
    }

    public final void asyncExecute(Runnable command) {
        this.workers.execute(command);
    }

    private synchronized SelectorStats getSelectorStats() {
        SelectorStats selectorStats = this.selectorStats;
        long currentTime = System.currentTimeMillis();
        if (currentTime < selectorStats.expireTime) {
            return selectorStats;
        }
        selectorStats = new SelectorStats();
        selectorStats.expireTime = currentTime + 1000L;
        for (SelectorThread selector : this.selectors) {
            selectorStats.operations += selector.operations;
            selectorStats.sessions += selector.sessions;
            selectorStats.maxReady = Math.max(selectorStats.maxReady, selector.maxReady);
        }
        this.selectorStats = selectorStats;
        return selectorStats;
    }

    private synchronized QueueStats getQueueStats() {
        QueueStats queueStats = this.queueStats;
        long currentTime = System.currentTimeMillis();
        if (currentTime < queueStats.expireTime) {
            return queueStats;
        }
        queueStats = new QueueStats();
        queueStats.expireTime = currentTime + 1000L;
        long[] stats = new long[2];
        for (SelectorThread selector : this.selectors) {
            for (Session session : selector.selector) {
                session.getQueueStats(stats);
                ++queueStats.sessions;
                queueStats.totalLength += stats[0];
                queueStats.totalBytes += stats[1];
                if (stats[0] > queueStats.maxLength) {
                    queueStats.maxLength = stats[0];
                }
                if (stats[1] <= queueStats.maxBytes) continue;
                queueStats.maxBytes = stats[1];
            }
        }
        this.queueStats = queueStats;
        return queueStats;
    }

    private static final class QueueStats {
        long expireTime;
        long totalLength;
        long totalBytes;
        long maxLength;
        long maxBytes;
        int sessions;

        private QueueStats() {
        }
    }

    private static final class SelectorStats {
        long expireTime;
        long operations;
        long sessions;
        int maxReady;

        private SelectorStats() {
        }
    }
}

