package k.common

import java.io.File
import java.net.URLDecoder
import java.util.jar.JarFile

const val namePrefix = "EXTENSION"
const val namePrefix_ = "EXTENSION_"
const val Unnamed = "Unnamed"

private const val CLASS_EXTENSIONS = "CLASS"
private val engines by lazy { appConfig.forceObj("Engines") }

private val Any.jarPath
    get() = URLDecoder.decode(this::class.java.protectionDomain.codeSource.location.path, "UTF-8").trim('/', '\\')

private val Any.jarEntries : List<String>
    get() {
        val path = jarPath

        return if (path.endsWith(".jar", true))
            JarFile(path).entries().asSequence().filter { !it.isDirectory }.map { it.str }.toList()
        else
            File(path).fileTree.map { it.str.substring(path.length + 1) }
    }

private val Any.extensionClasses
    get() = jarEntries
        .filter {
            it.endsWith(CLASS_EXTENSIONS, true) &&
                    it.contains(namePrefix, true) &&
                    !it.contains("$") &&
                    !it.endsWith("Kt")
        }
        .map { (it - ".$CLASS_EXTENSIONS").replace("\\", ".").replace("/", ".") }
        .mapNotNull { Class.forName(it) }

inline fun <reified T : Any> extensions() =
    Extensions(object {}, T::class.java)

class Extensions<T : Any>(refObject : Any, rootClass : Class<T>) : Map<String, Class<T>> {
    private val itemName = rootClass.className

    private val items = refObject.extensionClasses
        .filter { rootClass != it && rootClass.isAssignableFrom(it) }
        .map { it.cast<Class<T>>() }
        .associateBy { it.className }

    private val lowItems = items.entries.associate { it.key.low to it.value }

    override fun get(key : String) =
        lowItems[engines[key, key].low]
            ?: throw NotFoundError("""$itemName "$key" not found""")

    fun <R> apply(key : String, code : (extension : T) -> R) : R? {
        lowItems[key]?.let {
            return code(it())
        }

        return null
    }

    operator fun invoke(name : String) =
        this[name]()

    val desc
        get() = map { "${it.key.asCol(asText = true)} (${it.value().desc})" }.joinToString("\n")

    override val entries
        get() = items.entries

    override val keys
        get() = items.keys

    override val size =
        items.size

    override val values
        get() = items.values

    override fun isEmpty() =
        items.isEmpty()

    override fun containsValue(value : Class<T>) =
        items.containsValue(value)

    override fun containsKey(key : String) =
        key in lowItems

    operator fun contains(key : String) =
        lowItems.containsKey(key.low)

    val instances
        get() = items.map { it.value() }
}