package k.common

import java.io.File
import java.net.URI
import java.time.*
import java.util.*

/**
 * Return compact string representation of new UUID
 */
val ID
    get() = UUID.randomUUID().str - "-"

val zeroID = UUID(0, 0)

val Date.local : LocalDateTime
    get() = toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime()

operator fun <T> Collection<T>.contains(value : Collection<T>) =
    this.containsAll(value)

val Any?.bln
    get() = str.toBoolean()

@Suppress("UNCHECKED_CAST")
fun <T> Any?.cast() =
    this as T

val Any?.isNull
    get() = this == null

val Any?.isNotNull
    get() = this != null

fun <T> Boolean.choose(trueVal : T, falseVal : T) =
    if (this)
        trueVal
    else
        falseVal

inline infix fun <reified T> List<T>?.default(default : List<T>?) =
    this?.ifEmpty { default }
    ?: default

inline infix fun <reified T> List<T>?.default(default : T) =
    this?.ifEmpty { listOf(default) }
    ?: listOf(default)

inline infix fun <reified T> List<T>?.ensure(value : T) =
    if (this == null)
        listOf(value)
    else if (this contains value)
        this
    else
        this + value

infix fun <T> List<T>.plus(value : T?) : List<T> =
    if (value == null)
        this
    else
        this + value

/**
 * Append map if value is not null
 */
infix fun <K, V> Map<K, V>.plus(value : Pair<K, V>?) : Map<K, V> =
    if (value == null)
        this
    else
        this + value

/**
 * Append list if value is not null
 */
infix fun <T> List<T>?.contains(evaluator : (T) -> Boolean) =
    this?.find(evaluator) != null

inline infix fun <reified T> List<T>?.contains(value : T) =
    if (this == null)
        false
    else when (value)
    {
        is String -> this.find { it.cast<String>() same value } != null
        is File   -> this.find { (it.cast<File>().name - ".") same (value.name - ".") } != null
        else      -> this.contains(value)
    }

val URI.isEmpty
    get() = this.str.isBlank()

val isWindows
    get() = System.getProperty("os.name").startsWith("Windows", true)

infix fun <K, V> Map<K, V>.diff(value : Map<K, V>) =
    filter {
        val itemValue = value[it.key]
        itemValue == null || itemValue != it.value
    }