package k.parallels

import k.common.Duration
import java.util.concurrent.*

/**
 * Limit parallel execution of code with usual or virtual Threads without reentrant.
 */
class Limiter(count : Int = 1) {
    private val semaphore = Semaphore(count)

    fun lock() =
        semaphore.acquire()

    fun tryLock(timeOut : Duration) =
        semaphore.tryAcquire(timeOut.ms, TimeUnit.MILLISECONDS)

    fun unLock() =
        semaphore.release()

    inline operator fun <T> invoke(code : () -> T) : T {
        lock()

        try {
            return code()
        }
        finally {
            unLock()
        }
    }

    inline operator fun <T> invoke(timeOut : Duration, default : () -> T = { throw TimeoutException("Timeout $timeOut exceed") }, code : () -> T) =
        if (tryLock(timeOut))
            try {
                code()
            }
            finally {
                unLock()
            }
        else
            default()
}