package k.serializing

import k.common.*
import k.common.Duration
import k.ofl.*
import java.io.File
import java.lang.reflect.*
import java.net.*
import java.nio.file.Path
import java.text.SimpleDateFormat
import java.time.*
import java.time.format.DateTimeFormatter
import java.util.*
import kotlin.reflect.*

abstract class DeSerializer<S>(val source : String) {
    abstract fun getObj(obj : S, name : String) : S?
    abstract fun getValue(obj : S, path : String) : String?
    abstract fun getValue(item : Any) : String
    abstract fun getMap(obj : S, path : String) : Map<String, Any>?

    @OptIn(ExperimentalStdlibApi::class)
    inline fun <reified T : Any> deSerialize(obj : S, path : String) : T {
        val type = typeOf<T>().javaType

        val clazz = (if (type is ParameterizedType)
            type.rawType
        else
            type) as Class<*>

        return replaceError("Failed to unmarshall $clazz") {
            buildValue(path,
                       type,
                       clazz,
                       "",
                       obj,
                       "",
                       { source },
                       applyDefault = { value, _, _ -> value }
                      ).cast()
        }
    }

    fun buildList(obj : S, type : Type, clazz : Class<*>, path : String) =
        getMap(obj, path)
            ?.map { entry ->
                buildValue("",
                           type,
                           clazz,
                           "",
                           entry.value.cast(),
                           entry.key,
                           { getValue(entry.value) })
            }

    @Suppress("DEPRECATION")
    fun buildValue(path : String,
                   type : Type,
                   clazz : Class<*>,
                   format : String = "",
                   obj : S,
                   key : String,
                   getStr : () -> String,
                   applyDefault : (value : Any?, globalDefault : () -> Any?, allowDefValue : Boolean) -> Any? = { _, _, _ -> null }) : Any? =
        when (clazz) {
            String::class.java        -> getStr()
            OflObject::class.java,
            OFL::class.java           -> OFL(getStr())

            Byte::class.java          -> getStr().toUByte()
            UByte::class.java         -> getStr().toUByte()
            Short::class.java         -> getStr().toShort()
            UShort::class.java        -> getStr().toUShort()
            Int::class.java,
            Integer::class.java       -> getStr().toInt()

            UInt::class.java          -> getStr().toUInt()
            Long::class.java          -> getStr().toLong()
            ULong::class.java         -> getStr().toULong()
            Double::class.java        -> getStr().toDouble()
            Float::class.java         -> getStr().toFloat()
            Duration::class.java      -> getStr().duration
            Boolean::class.java       -> getStr().toBoolean()
            LocalDateTime::class.java -> LocalDateTime.parse(getStr())
            LocalDate::class.java     -> LocalDate.parse(getStr())

            Date::class.java          -> when (format) {
                ""               -> Date(getStr().long)
                "ISO8601"        -> SimpleDateFormat("yyyy-MM-dd HH:mm:ss Z z").parse(getStr())
                "ISO8601-OFFSET" -> Date(OffsetDateTime.parse(getStr(), DateTimeFormatter.ISO_OFFSET_DATE_TIME).toInstant().toEpochMilli())

                else             -> Date.parse(getStr())
            }

            Instant::class.java       -> Instant.parse(getStr())
            URI::class.java           -> URI(getStr())
            URL::class.java           -> URI(getStr()).toURL()
            Path::class.java          -> Path.of(getStr())
            File::class.java          -> File(getStr())

            UUID::class.java          ->
                when (val valueStr = getStr()) {
                    ""   -> UUID.randomUUID()
                    "0"  -> zeroID

                    else ->
                        UUID.fromString(valueStr)
                }

            LinkedList::class.java    ->
                LinkedList(applyDefault(buildList(obj, type, type[0], path),
                                        { listOf<Any>() },
                                        false).cast<List<Any>>()
                          )

            ArrayList::class.java,
            List::class.java          ->
                applyDefault(buildList(obj, type, type[0], path),
                             { listOf<Any>() },
                             false)

            // Todo: Map<String, List<>>
            Map::class.java           -> {
                applyDefault(getMap(obj, path)?.mapValues { entry ->
                    buildValue(path,
                               clazz,
                               type[1],
                               "",
                               entry.value.cast(),
                               entry.key,
                               { getValue(entry.value) })
                },
                             { mapOf<String, Any>() },
                             false)
            }

            // Todo: List<Pair<>>
            /*  Pair::class.java          -> applyDefault(getList(path),
                                                    { listOf<Any>() },
                                                    false)
              .cast<List<String>>()
              .map {
                  val parts = it.split(":")

                  Pair(buildValue(path, clazz, type[0], format, { parts[0] }, applyDefault),
                       buildValue(path, clazz, type[1], format, { parts[1] }, applyDefault))
              }*/

            else                      -> if (clazz.isEnum) {
                val valueStr = getStr()

                clazz.enumConstants
                    .firstOrNull { it.str same valueStr } orThrow "$valueStr is not part of $clazz"
            }
            else {
                val childObj = if (path == "")
                    obj
                else
                    getObj(obj, path)

                if (childObj == null)
                    applyDefault(null, { clazz() }, true)
                else
                    buildObject(clazz, childObj.cast(), path, key)
            }
        }

    private fun buildObject(clazz : Class<*>,
                            obj : S,
                            path : String,
                            key : String) : Any {
        var keyValue = key

        val args = clazz.declaredFields
            .take(clazz.constructors.maxOf { it.parameterCount }) // TODO: Нужно?
            .filter { it.annotation<Hide>().isNull }
            .map { field ->
                val fieldName = field.annotation<Name>()?.value default field.name
                val fullPath = "${path.str merge "."}$fieldName"

                val res = buildValue(fieldName,
                                     field.genericType,
                                     field.type,
                                     field.annotation<Format>()?.value ?: "",
                                     obj,
                                     "",
                                     getStr = {
                                         keyValue.ifEmpty { getValue(obj, fieldName) ?: (field.default mustBeFound fullPath) }
                                     },
                                     applyDefault = { value, noValueDefault, allowDefValue ->
                                         value
                                             ?: run {
                                                 val defValue = field.default mustBeFound path

                                                 if (defValue.isEmpty())
                                                     noValueDefault()
                                                 else if (allowDefValue)
                                                     defValue
                                                 else
                                                     throw CheckError("Default value for $path must be empty string")
                                             }
                                     })

                keyValue = ""
                res!!
            }

        return clazz.create(args.toTypedArray())
    }

    operator fun Type.get(index : Int) =
        (this as ParameterizedType).actualTypeArguments[index] as Class<*>

    val Field.default
        get() = annotation<Optional>()?.value
}