package k.serializing

import k.common.notImplemented
import k.serializing.json.*
import k.stream.text
import java.io.InputStream
import java.util.*

inline fun <reified T : Any> InputStream.deSerialize() : T =
    this.text.deSerialize()

inline fun <reified T : Any> String.deSerialize() : T =
    object : DeSerializer<JsonNode>(this) {
        override fun getObj(obj : JsonNode, name : String) =
            (obj as JsonObject)[name] as? JsonContainer

        override fun getValue(obj : JsonNode, path : String) =
            ((obj as JsonObject)[path] as? JsonProperty)?.value

        override fun getValue(item : Any) =
            when (item) {
                is JsonProperty -> item.value
                is JsonNode     -> item.name
                else            -> error("Unexpected item type ${item.javaClass}")
            }

        override fun getMap(obj : JsonNode, path : String) : Map<String, JsonNode>? {
            val array = if (path.isEmpty())
                obj
            else
                (obj as JsonObject)[path]

            return (array as? JsonArray)
                ?.items
                ?.associateBy { it.key }
        }
    }.deSerialize(JsonNode.parse(this), "")

fun <T : Any> T?.serialize() : String {
    if (this == null)
        return "{}"

    when (this) {
        is List<*> ->
            return joinToString(",", "[", "]") { it.serialize() }

        else       -> {}
    }

    val obj = JsonObj(StringBuilder(), Stack<Char>())
    val list = JsonList(obj.builder, obj.stack)

    obj.obj = obj
    obj.list = list

    list.obj = obj
    list.list = list

    obj.builder.append(obj.startChar)

    JsonSerializer().serialize(obj, this, false, null)

    obj.close(0)

    obj.builder.append(obj.finishChar)

    return obj.builder.toString()
}

const val divider = ','

open class JsonObj(val builder : StringBuilder,
                   val stack : Stack<Char>) {
    open val startChar = '{'
    open val finishChar = '}'

    lateinit var list : JsonObj
    lateinit var obj : JsonObj

    private fun isNumber(value : String) : Boolean {
        var hasDot = 0

        value.forEach { char ->
            when (char) {
                '.', ','                                         -> hasDot++
                '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' -> {}

                else                                             -> return false
            }
        }

        return hasDot < 2
    }

    fun setValue(name : String, value : String, level : Int) {
        start(name, level)

        val quoted = when (value.firstOrNull()) {
            '0', '1', '2', '3', '4', '5', '6', '7', '8', '9' -> !isNumber(value)
            't'                                              -> value != "true"
            'f'                                              -> value != "false"
            'n'                                              -> value != "null"

            else                                             -> true
        }

        if (quoted)
            builder.append('"')

        value.forEach { char ->
            when (char) {
                '\"'     -> builder.append("\\\"")
                '\\'     -> builder.append("\\\\")
                '\b'     -> builder.append("\\b")
                '\u000C' -> builder.append("\\f")
                '\n'     -> builder.append("\\n")
                '\r'     -> builder.append("\\r")
                '\t'     -> builder.append("\\t")
                else     -> {
                    val codePoint = char.code

                    if (codePoint in 0x0..0x1F || codePoint in 0x7F..0x9F)
                        builder.append("\\u%04X".format(codePoint))
                    else
                        builder.append(char)
                }
            }
        }

        if (quoted)
            builder.append('"')
    }

    private fun divide() {
        if (builder.last() !in "{[")
            builder.append(divider)
    }

    private fun start(name : String, level : Int) {
        close(level)
        divide()

        if (name.isNotEmpty())
            builder.append("\"$name\":")
    }

    fun startObj(name : String, level : Int) {
        start(name, level)

        builder.append(startChar)

        stack.push(finishChar)
    }

    fun close(level : Int) {
        while (obj.stack.size > level)
            obj.builder.append(obj.stack.pop())
    }
}

class JsonList(builder : StringBuilder, stack : Stack<Char>) : JsonObj(builder, stack) {
    override val startChar = '['
    override val finishChar = ']'
}

class JsonSerializer : Serializer<JsonObj>() {
    override fun getObj(root : JsonObj, path : String) =
        root.obj.also {
            it.startObj(path, level)
        }

    override fun getList(root : JsonObj, path : String) =
        root.list.also {
            it.startObj(path, level)
        }

    override fun setValue(obj : JsonObj, name : String, value : String) =
        obj.setValue(name, value, level)

    override fun isNativeValue(value : Any) =
        false

    override fun setNativeValue(name : String, obj : JsonObj, value : Any) : Unit =
        notImplemented()
}