package k.stream

import k.common.*
import java.io.*
import java.lang.Thread.startVirtualThread
import java.nio.ByteBuffer
import java.util.*
import java.util.zip.*

val InputStream.long
    get() = ByteBuffer.wrap(readNBytes(Long.SIZE_BYTES)).getLong()

val InputStream.longs
    get() = int.map { long }

val InputStream.int
    get() = ByteBuffer.wrap(readNBytes(Int.SIZE_BYTES)).getInt()

val InputStream.ints
    get() = int.map { int }

val InputStream.short
    get() = ByteBuffer.wrap(readNBytes(Short.SIZE_BYTES)).getShort()

val InputStream.dbl
    get() = ByteBuffer.wrap(readNBytes(Double.SIZE_BYTES)).getDouble()

val InputStream.byte
    get() = readNBytes(Byte.SIZE_BYTES)[0]

val InputStream.bln
    get() = byte.bln

val InputStream.array
    get() = readNBytes(int)

val InputStream.string
    get() = readNBytes(int).toString(Charsets.UTF_8)

val InputStream.uuid
    get() = UUID(long, long)

val InputStream.uuids
    get() = int.map { uuid }

val InputStream.time : Date
    get() = Date(long)

val InputStream.unCompressed
    get() = let {
        if (markSupported())
            reset()

        GZIPInputStream(this)
    }

val InputStream.compressedStream
    get() = PipedInputStream()
        .also {
            if (markSupported())
                reset()

            val output = PipedOutputStream(it)
            val zip = GZIPOutputStream(output)

            startVirtualThread {
                output.use {
                    copyTo(zip)
                    zip.finish()
                }
            }
        }

val InputStream.compressed : ByteArray
    get()
    {
        val buf = ByteArrayOutputStream()
        val zip = GZIPOutputStream(buf)
        zip.write(readAllBytes())
        zip.finish()
        return buf.toByteArray()
    }

val InputStream.text
    get() = reader().readText()