package k.common

import java.text.DecimalFormat
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.math.abs
import kotlin.system.measureNanoTime

const val nanoSecond = 1L
const val microSecond = nanoSecond * Thousand
const val milliSecond = microSecond * Thousand
const val second = milliSecond * Thousand
const val minute = second * 60
const val hour = minute * 60
const val day = hour * 24
const val week = day * 7
const val month = day * 30
const val year = day * 365

/**
 * Time duration with nanoseconds precision
 */

class Duration(val value : Long) {
    constructor(str : String) : this(parseDuration(str))
    constructor(code : () -> Unit) : this(measureNanoTime { code() })
    constructor(date : Date) : this(Date(), date)
    constructor(someTime : Date, anotherTime : Date) : this(TimeUnit.MILLISECONDS.toNanos(abs(someTime.time - anotherTime.time)))

    val us
        get() = value / microSecond

    val ms
        get() = value / milliSecond

    val sec
        get() = value / second

    val min
        get() = value / minute

    val hours
        get() = value / hour

    val days
        get() = value / day

    val months
        get() = value / month

    val isZero
        get() = value == 0L

    val isPositive
        get() = value > 0L

    val isNegative
        get() = value < 0L

    operator fun minus(duration : Duration) =
        Duration(value - duration.value)

    operator fun plus(time : Date) =
        Date(ms + time.time)

    operator fun plus(duration : Duration) =
        Duration(value + duration.value)

    operator fun plus(duration : Long) =
        Duration(value + duration)

    operator fun div(divider : Long) =
        Duration(value / divider)

    operator fun div(divider : Double) =
        Duration((value / divider).toLong())

    operator fun div(divider : Int) =
        Duration(value / divider)

    operator fun div(duration : Duration) =
        value.toDouble() / duration.value

    operator fun times(multiplicator : Long) =
        Duration(value * multiplicator)

    operator fun times(multiplicator : Int) =
        Duration(value * multiplicator)

    operator fun times(multiplicator : Double) =
        Duration((value * multiplicator).long)

    operator fun compareTo(duration : Duration) =
        if (duration.value == value)
            0
        else if (duration.value > value)
            -1
        else
            1

    operator fun compareTo(duration : Long) =
        if (duration == value)
            0
        else if (duration > value)
            -1
        else
            1

    operator fun compareTo(duration : String) =
        compareTo(Duration(duration))

    override operator fun equals(other : Any?) =
        when (other) {
            is Long     -> other == value
            is Int      -> other.toLong() == value
            is Byte     -> other.toLong() == value
            is Duration -> other.value == value

            else        -> false
        }

    infix fun max(value : Duration) =
        if (this.value > value.value)
            this
        else
            value

    infix fun min(value : Duration) =
        if (this.value < value.value)
            this
        else
            value

    companion object {
        val ZERO = Duration(0L)
        val INFINITE = Duration(Long.MAX_VALUE)
        val NEGATIVE_INFINITE = Duration(Long.MIN_VALUE)
        val ATOM = 1L
    }

    override fun toString() =
        str

    val str
        get() = value.durationStr(true)

    val string
        get() = value.durationStr(false)

    val duration : java.time.Duration
        get() = java.time.Duration.ofNanos(value)

    /**
     * Freeze the current stream for the time determined by the current Duration
     */
    fun sleep() {
        if (ms >= 1)
            Thread.sleep(ms)
    }

    override fun hashCode() =
        value.hashCode()

    /**
     * Random Duration from 0 to the current value
     */
    val random
        get() = Duration((0L..value).random())
}

fun Long.durationStr(short : Boolean) : String {
    if (this == 0L)
        return "0"

    var v = this
    var res = ""

    arrayOf(month to "mon",
            day to "d",
            hour to "h",
            minute to "m",
            second to "s",
            milliSecond to "ms",
            microSecond to "us",
            nanoSecond to "ns")
        .forEach {
            val part = v.toDouble() / it.first.toLong()
            val truncVal = part.toLong()
            v -= truncVal * it.first.toLong()

            if (!truncVal.isZero) {
                if (short)
                    return DecimalFormat(if (part < 10) "#.#" else "#").format(part) + " " + it.second

                res += "$truncVal ${it.second} "
            }
        }

    return res.trimEnd()
}

fun parseDuration(str : String) =
    replaceError("Invalid duration string") {
        var duration = 0L
        var fixedStr = str.low - " " - "," - "."

        fun parsePart(multiplicator : Long, vararg suffixes : String) {
            suffixes.forEach {
                if (fixedStr.contains(it)) {
                    val parts = fixedStr.split(it)

                    if (parts.first().isNotBlank() && parts.first().last().isDigit() && (parts[1].isEmpty() || parts[1].first().isDigit())) {
                        fixedStr = parts[1]
                        duration += parts.first().long * multiplicator

                        return
                    }
                }
            }
        }

        parsePart(month, "months", "month", "mon")
        parsePart(week, "weeks", "week", "w")
        parsePart(day, "days", "day", "d")
        parsePart(hour, "hours", "hour", "hr", "h")
        parsePart(minute, "min", "m")
        parsePart(second, "sec", "s")
        parsePart(milliSecond, "ms")
        parsePart(microSecond, "us", "u")
        parsePart(nanoSecond, "ns", "n")

        if (fixedStr.isNotBlank())
            duration += fixedStr.long

        duration
    }

val Int.h
    get() = Duration(this * hour)

val Int.min
    get() = Duration(this * minute)

val Int.sec
    get() = Duration(this * second)

val Int.ms
    get() = Duration(this * milliSecond)

val Int.us
    get() = Duration(this * microSecond)

val Long.h
    get() = Duration(this * hour)

val Long.min
    get() = Duration(this * minute)

val Long.sec
    get() = Duration(this * second)

val Long.ms
    get() = Duration(this * milliSecond)

val Long.us
    get() = Duration(this * microSecond)

val Double.sec
    get() = Duration((this * second).toLong())

val Double.min
    get() = Duration((this * minute).toLong())

val Double.h
    get() = Duration((this * hour).toLong())