package k.common

import java.io.InputStream
import java.nio.charset.StandardCharsets
import java.text.DecimalFormat
import java.util.*
import kotlin.math.*

const val HundredPercent = "100%"

fun Char.pad(count : Int) =
    "".padStart(0.coerceAtLeast(count), this)

fun String.asCol(width : Int = 10, asText : Boolean = false) : String {
    val padLen = width - (this - ANSI_COLOR_PATTERN).length

    return if (asText)
        this + ' '.pad(padLen)
    else
        ' '.pad(padLen - 1) + "$this "
}

operator fun String.minus(value : String) =
    this.replace(value, "", true)

operator fun String.minus(value : Regex) =
    this.replace(value, "")

val String.title
    get() = isNotEmpty() accept this[0].uppercase() + substring(1).low

val String.unTitle
    get() = isNotEmpty() accept this[0].lowercase() + substring(1)

fun String.suffix(value : String) =
    if (isEmpty() || substring(length - value.length) == value) this else this + value

fun String.prefix(value : String) =
    if (length >= value.length && substring(0, value.length) == value) this else value + this

val String.dot
    get() = suffix(".")

val String.s
    get() = suffix("s")

inline val String.byte
    get() = dbl.byte

inline val String.int
    get() = dbl.int

inline val String.long
    get() = dbl.long

//val Any.int: Int
//    get() = (this as String).int

inline val String.dbl
    get() = this.trim { it !in "-0123456789." }.replace(',', '.').toDouble()

val String.bln
    get() = toBoolean()

val String.uuid : UUID
    get() = UUID.fromString(this)

val String.duration : Duration
    get() = Duration(this)

val String.low
    get() = lowercase(Locale.getDefault())

val String.up
    get() = uppercase(Locale.getDefault())

infix fun String.hasOneOf(chars : String) : Boolean {
    chars.forEach { if (it in this) return true }

    return false
}

val String.n
    get() = this + "\n"

fun String.quote(char : Char = '"') =
    "$char$this$char"

infix fun String.same(text : String) : Boolean =
    equals(text, true)

val String.asIdent
    get() = this.low.replace("\\s".toRegex(), "_")

val String.maskRegExp : String
    get() = this.map { if (it in """\.*[]-()?""") """\$it""" else it }.joinToString("")

val String.urlDecode : String
    get() = java.net.URLDecoder.decode(this, StandardCharsets.UTF_8)

val String.print : String
    get() = this.also { msg(this.n, MsgType.Warning) }

infix fun Boolean?.accept(value : String) =
    if (this == true)
        value
    else
        ""

infix fun Boolean?.accept(value : () -> String) =
    if (this == true)
        value()
    else
        ""

infix fun Any?.accept(value : String) =
    if (this != null)
        value
    else
        ""

infix fun String.percentOf(value : Int) =
    value percentOf this.dbl

val String.base64 : String
    get() = Base64.getEncoder().encodeToString(this.toByteArray(StandardCharsets.UTF_8))

val String.deBase64 : String
    get() = Base64.getDecoder().decode(this).decodeToString()

val String.deBase64ToStream : InputStream
    get() = Base64.getDecoder().decode(this).inputStream()

val Double.percentStr
    get() = DecimalFormat(when {
                              this < 1  -> "#.##"
                              this < 10 -> "#.#"
                              else      -> "#"
                          }
                         ).format(this) + " %"

infix fun String?.default(value : String) =
    if (this.isNullOrEmpty())
        value
    else
        this

infix fun String?.default(value : () -> String) =
    if (this.isNullOrEmpty())
        value()
    else
        this

infix fun String.and(value : String?) =
    if (value.isNullOrBlank())
        value.str
    else
        "$this$value"

infix fun String?.merge(value : String?) =
    if (isNullOrBlank())
        ""
    else
        "$this$value"

infix fun String?.accept(value : String) =
    if (isNullOrEmpty())
        ""
    else
        value

infix fun String.extract(expr : String) =
    expr.toRegex(RegexOption.IGNORE_CASE).find(this)?.groupValues?.get(1)

infix fun String.pairBy(divider : String) =
    substringBefore(divider) to substringAfter(divider)

infix fun <T> String?.acceptList(list : () -> List<T>) : List<T> =
    if (this.isNullOrBlank())
        listOf()
    else
        list()

infix fun String?.accept(code : (String) -> Unit) {
    if (!isNullOrBlank())
        code(this)
}

infix fun MutableList<String>.addSignificant(value : String?) {
    if (!value.isNullOrBlank())
        add(value)
}

val String.list
    get() = this.split(',', '\n').map { it.trim() }.filter { it.isNotBlank() }

fun String.sub(start : Int, finish : Int) : String {
    val startPos = min(max(start, 0), length)
    val finishPos = max(min(finish, length), 0)

    return substring(startPos, finishPos)
}

val String.quoted
    get() = if (contains("\""))
        this
    else
        "\"$this\""