package k.ofl

import k.common.*
import k.ofl.*

const val varMarker = '$'
const val varMask = '\\'
const val varBodyStart = '{'
const val varBodyFinish = '}'

fun OflItem.eval(expr : String, vararg externalData : OflObject) : String
{
    var intermediate = expr

    while (true)
    {
        val match = extractVarName(intermediate)

        if (match.name.isBlank())
            break

        val context = if (this is OflObject)
            this
        else
            this.parent mustBeFound "Invalid OFL tree"

        val item = context.getOrNull(match.name, OflObject.OflRecursive.Up)
                   ?: (externalData.firstNotNullOfOrNull { it.getOrNull(match.name) } mustBeFound match.name)

        intermediate = intermediate.sub(0, match.start) +
                       eval(item.value, *externalData) +
                       intermediate.sub(match.finish + 1, intermediate.length)
    }

    return intermediate.replace("\\\$", "$")
}

private class VarMatch(val start : Int,
                       val finish : Int,
                       val name : String)

private fun extractVarName(text : String) : VarMatch
{
    val lastPos = text.length - 1
    var startPos = lastPos

    while (startPos >= 0 && (text[startPos] != varMarker || (startPos > 0 && text[startPos - 1] == varMask)))
        startPos--

    if (startPos < 0 || text.length < 2)
        return VarMatch(0, 0, "")

    if (text[startPos + 1] == varBodyStart)
    {
        val startBodyPos = startPos + 2

        for (finishBodyPos in startBodyPos..lastPos)
            if (text[finishBodyPos] == varBodyFinish)
                return VarMatch(startPos, finishBodyPos, text.substring(startBodyPos, finishBodyPos))

        throw OFLError("Unterminated variable")
    }
    else
    {
        val startBodyPos = startPos + 1

        for (finishBodyPos in startBodyPos..lastPos)
            if (!text[finishBodyPos].isLetter())
                if (startBodyPos == finishBodyPos)
                    throw OFLError("Variable without name")
                else
                    return VarMatch(startPos, finishBodyPos - 1, text.substring(startBodyPos, finishBodyPos))

        return VarMatch(startPos, lastPos, text.substring(startBodyPos, lastPos + 1))
    }
}