package k.parallels

import k.common.Duration
import java.util.concurrent.*
import java.util.concurrent.locks.ReentrantReadWriteLock

/**
 * Ensure execution of code with only ONE usual or virtual Thread with reentrant.
 */
class Sync(fair : Boolean = false) {
    private val locker = ReentrantReadWriteLock(fair)

    fun lock() =
        locker.writeLock().lock()

    fun tryLock(timeOut : Duration) =
        locker.writeLock().tryLock(timeOut.ms, TimeUnit.MILLISECONDS)

    fun unLock() =
        locker.writeLock().unlock()

    fun readLock() =
        locker.readLock().lock()

    fun tryReadLock(timeOut : Duration) =
        locker.readLock().tryLock(timeOut.ms, TimeUnit.MILLISECONDS)

    fun readUnLock() =
        locker.readLock().unlock()

    /**
     * Execute code with read lock (synchronization ONLY with write locks)
     */
    inline fun <T> read(code : () -> T) : T {
        readLock()

        try {
            return code()
        }
        finally {
            readUnLock()
        }
    }

    /**
     * Execute code with read lock (synchronization ONLY with write locks) and check timeout
     */
    inline fun <T> read(timeOut : Duration, default : () -> T = { throw TimeoutException("Read timeout $timeOut exceed") }, code : () -> T) =
        if (tryReadLock(timeOut))
            try {
                code()
            }
            finally {
                readUnLock()
            }
        else
            default()

    /**
     * Execute code with write lock (full synchronization)
     */
    inline operator fun <T> invoke(code : () -> T) : T {
        lock()

        try {
            return code()
        }
        finally {
            unLock()
        }
    }

    /**
     * Execute code with write lock (full synchronization) and check timeout
     */
    inline operator fun <T> invoke(timeOut : Duration, default : () -> T = { throw TimeoutException("Timeout $timeOut exceed") }, code : () -> T) =
        if (tryLock(timeOut))
            try {
                code()
            }
            finally {
                unLock()
            }
        else
            default()
}