package k.common

enum class MsgType(val color : AnsiColor, val levels : String)
{
    Debug(AnsiColor.Green, "d"),
    Info(AnsiColor.Ocean, "di"),
    InfoOk(AnsiColor.Green, "di"),
    Text(AnsiColor.White, "din"),
    BlueText(AnsiColor.Blue, "din"),
    OrangeText(AnsiColor.Orange, "din"),
    Disabled(AnsiColor.Gray, "din"),
    Ok(AnsiColor.Green, "din"),
    Warning(AnsiColor.Orange, "dwin"),
    Error(AnsiColor.Red, "dwein")
}

const val ANSI_COMMAND = "\u001b["
const val ANSI_COLOR_SUFFIX = "m"
val ANSI_COLOR_PATTERN = Regex("\u001b\\[.*?m")
const val resetConFormat = ANSI_COMMAND + ANSI_COLOR_SUFFIX
const val ANSI_COLOR_TEXT_OFFSET = 30
const val ANSI_COLOR_BACK_OFFSET = 40

enum class AnsiColor(val value : Int)
{
    Black(0),
    Red(1),
    Green(2),
    Orange(3),
    Blue(4),
    Violet(5),
    Ocean(6),
    Gray(7),
    DarkGray(60),
    Pink(61),
    Lime(62),
    Yellow(63),
    Sky(64),
    LightViolet(65),
    LightOcean(66),
    White(67),
    Default(9),
    Undefined(-1)
    {
        override val text = ""
        override val back = ""
    };

    infix fun or(color : AnsiColor) =
        if (this == Undefined) color else this

    open val text : String
        get() = ";" + (ANSI_COLOR_TEXT_OFFSET + value).str

    open val back : String
        get() = ";" + (ANSI_COLOR_BACK_OFFSET + value).str

    operator fun invoke(text : String) =
        text.conFormat(this)
}

fun String.conFormat(textColor : AnsiColor = AnsiColor.White, backColor : AnsiColor = AnsiColor.Undefined) =
    ANSI_COMMAND + (textColor.text + backColor.back).trim(';') + ANSI_COLOR_SUFFIX + this

class Messages(private val code : Messages.() -> Unit = {})
{
    private var offset = 0
    private val levelOffset = 4

    init
    {
        code(this)
    }

    fun <T> level(code : () -> T) : T
    {
        try
        {
            offset += levelOffset

            return code()
        }
        finally
        {
            offset -= levelOffset
        }
    }

    fun msg(message : String = "\n", type : MsgType = MsgType.Text) : Int
    {
        if (logging in type.levels)
            msgInt(message, type)

        return message.length
    }

    val offsetStr
        get() = ' '.pad(offset)
}

private val logging by future { appConfig["logging", "n"].first() }

fun msg(message : String = "\n", type : MsgType = MsgType.Text) : Int
{
    if (logging in type.levels)
        msgInt(message, type)

    return message.length
}

fun msgInt(message : String, type : MsgType = MsgType.Info, backColor : AnsiColor = AnsiColor.Undefined)
{
    print(message.conFormat(type.color, backColor))
    print(resetConFormat)
}

fun debug(block : () -> Unit)
{
    if (logging == 'd')
        block()
}