package k.common

import org.slf4j.Logger
import java.util.concurrent.CopyOnWriteArrayList

fun softTryProc(timeout : Duration = 5.sec, delay : Duration = 1.sec, proc : () -> Unit) : Boolean {
    val timer = Timer()

    while (true) {
        try {
            proc()

            return true
        }
        catch (_ : Throwable) {
            if (timer.time > timeout)
                return false

            delay.sleep()
        }
    }
}

fun <T> tryProc(timeout : Duration = 5.sec, delay : Duration = 1.sec, proc : () -> T) : T {
    val timer = Timer()

    while (true) {
        try {
            return proc()
        }
        catch (e : Throwable) {
            if (timer.time > timeout)
                throw e

            delay.sleep()
        }
    }
}

fun perfCheck(label : String = "", code : () -> Unit) =
    Duration { code() }.also { println("$label${it.str}") }

fun perfTest(limit : Duration = Duration.ZERO, warming : Boolean = false, showResults : Boolean = true, code : () -> Unit) : Duration {
    val fastMS = 5.ms

    if (warming || (!limit.isZero && limit < fastMS))
        perfTest(showResults = false, code = code)

    var minCount = 10
    val fastCount = 100

    var totalDuration = Duration.ZERO
    var maxDuration = Duration.NEGATIVE_INFINITE
    var minDuration = Duration.INFINITE
    var seriesCount = 0
    var totalCount = 0

    while (seriesCount < minCount) {
        val duration = Duration { code() }

        totalCount++
        seriesCount++

        if (duration < fastMS) {
            minCount = fastCount
        }

        if (duration > maxDuration) {
            maxDuration = duration
            seriesCount = 0
        }

        if (duration < minDuration) {
            minDuration = duration
            seriesCount = 0
        }

        totalDuration += duration
    }

    val avgDuration = totalDuration / totalCount

    if (showResults) {
        msgInt("\n==== Performance test ====\n\n", MsgType.Text)

        if (!warming && avgDuration < fastMS) {
            msgInt("The result is unstable because the average duration is less than ${fastMS.str}\n",
                   MsgType.Warning)
        }

        msgInt("Avg: ${avgDuration.str}", MsgType.Info)

        if (limit > 0 && limit >= avgDuration) {
            msgInt(" [$limit passed]", MsgType.Ok)
        }

        msgInt("\nMin: $minDuration\n", MsgType.Ok)
        msgInt("Max: $maxDuration\n\n", MsgType.Error)

        msgInt("Count : $totalCount\n", MsgType.Text)

        msgInt("\n==========================\n\n", MsgType.Text)
    }

    (limit.isZero || limit >= avgDuration) orThrow "Time limit ($limit) exceeded with a value of $avgDuration!"

    return avgDuration
}

fun <T> stage(message : String, msgType : MsgType = MsgType.Info, logError : Boolean = false, errorText : String = "", code : () -> T) : T {
    try {
        val timer = Timer()

        msg("$message... ".n, msgType)

        return code().also {
            msg("Done (${timer.time})".n, MsgType.InfoOk)
        }
    }
    catch (e : Throwable) {
        val text = message.unTitle.dot.prefix("Failed to ")
        val msg = errorText.ifBlank { "\n$text\n${e.str.trim()}" }.n.replace("\n", "\n    ")

        if (logError) {
            msg(msg, MsgType.Error)

            e.printStackTrace()
        }

        throw Error(msg)
    }
}

fun <T> stage(message : String, log : Logger, errors : CopyOnWriteArrayList<String>? = null, errorText : String = "", code : () -> T) : T {
    try {
        message orThrow "Empty stage message"

        val timer = Timer()

        log.info("$message... ")

        return code().also {
            log.info("$message: Done (${timer.time})")
        }
    }
    catch (e : Throwable) {
        val text = message.unTitle.dot.prefix("Failed to ")
        val msg = (errorText default { "\n$text\n${e.str.trim()}" }).n.replace("\n", "\n    ")

        log.error(msg)

        errors?.add(msg)

        throw Error(msg)
    }
}

inline fun <T> mute(code : () -> T) : T? =
    try {
        code()
    }
    catch (e : Throwable) {
        when (e) {
            is OutOfMemoryError                     -> throw e
            is UninitializedPropertyAccessException -> throw e
            else                                    ->
                null
        }
    }

inline fun <T> mute(default : (e : Throwable) -> T, code : () -> T) : T =
    try {
        code()
    }
    catch (e : Throwable) {
        when (e) {
            is OutOfMemoryError                     -> throw e
            is UninitializedPropertyAccessException -> throw e
            else                                    ->
                default(e)
        }
    }

fun softCatch(code : () -> Unit) =
    try {
        code()

        ""
    }
    catch (e : Throwable) {
        e.str.trim()
    }