package k.stream

import k.common.*
import java.io.*
import java.lang.Integer.min
import java.lang.Long.min
import java.nio.ByteBuffer
import java.nio.charset.Charset
import java.util.*

const val copyPiece = 1 * KB.toInt()

open class Stream(private val storage : StreamStorage = MemoryStorage()) : InputStream() {
    constructor(source : InputStream) : this() {
        copyFrom(source)
        first()
    }

    val mirror
        get() = Stream(storage)

    var size
        get() = storage.size
        set(value) {
            storage.size = value

            if (position > value)
                position = value
        }

    val isEmpty
        get() = size == 0L

    fun first() =
        this.also {
            position = 0
        }

    fun last() =
        this.also {
            position = size
        }

    fun clear() {
        size = 0
    }

    var position = 0L
        set(value) {
            if (value > size)
                size = value

            field = value
        }

    private var mark = 0L

    val remainSize
        get() = size - position

    fun copyFrom(source : InputStream, copySize : Long = -1) {
        var transferSize = if (copySize < 0)
            source.available().long
        else
            copySize

        val needSize = transferSize + position

        if (needSize > size)
            size = needSize

        val buf = ByteBuffer.allocate(min(copyPiece, transferSize.int))

        while (transferSize > 0) {
            buf.limit(min(buf.capacity().long, transferSize).int)

            source.read(buf.array())
            write(buf)

            transferSize -= buf.limit()
        }
    }

    var string : String
        get() = read(int).array().toString(Charsets.UTF_8)
        set(value) {
            val buf = ByteBuffer.wrap(value.toByteArray())
            int = buf.limit()
            write(buf)
        }

    fun read(size : Int) =
        ByteBuffer.allocate(size)?.also { buf ->
            read(buf)
        }
            ?: throw OutOfMemoryError()

    fun read(value : ByteBuffer) {
        if (value.limit() == 0)
            return

        (value.limit() <= remainSize) orThrow "Not enough data"

        storage.read(value.position(0), position)

        position += value.limit()
    }

    val eof
        get() = size <= position

    fun write(value : ByteArray, size : Int = -1) =
        write(ByteBuffer.wrap(value).limit(size default value.size))

    //
    fun write(value : ByteBuffer, entireContent : Boolean = true) {
        if (value.limit() == 0)
            return

        val offset = entireContent.choose(0, value.position())
        val newSize = position + value.limit() - offset

        if (newSize > size)
            size = newSize

        storage.write(value.position(offset), position)

        position = newSize
    }

    fun writeStrBuf(value : String, charset : Charset = Charsets.UTF_8) =
        write(ByteBuffer.wrap(value.toByteArray(charset)))

    fun writeCSV(vararg value : Any) =
        writeStrBuf(value.joinToString(",").n)

    fun readStrBuf(size : Int = this.size.int) =
        read(size).array().toString(Charsets.UTF_8)

    fun flush() =
        storage.flush()

    override fun read() =
        if (remainSize > 0)
            byte.toInt()
        else
            -1

    private fun realLen(size : Int) =
        min(size, remainSize.toInt())

    override fun read(b : ByteArray, off : Int, len : Int) : Int {
        val realLen = realLen(len - off)

        if (realLen == 0)
            return -1

        System.arraycopy(read(realLen).array(), 0, b, off, realLen)

        return realLen
    }

    override fun readAllBytes() =
        ByteArray(size.toInt()).also {
            read(it, 0, it.size)
        }

    override fun transferTo(out : OutputStream) : Long {
        throw NotImplementedError()
    }

    override fun skip(n : Long) : Long {
        position += n
        return n
    }

    override fun available() =
        remainSize.toInt()

    override fun markSupported() =
        true

    override fun mark(readAheadLimit : Int) {
        mark = position
    }

    override fun reset() {
        position = mark
    }

    override fun close() {
        storage.close()
    }

    override fun toString() =
        storage.className + (storage.size < Long.MAX_VALUE).accept(": ${position.sizeStr} of ${size.sizeStr}")

    override fun equals(other : Any?) : Boolean {
        if (other !is Stream || size != other.size)
            return false

        val aBuf = ByteBuffer.allocate(copyPiece)
        val bBuf = ByteBuffer.allocate(copyPiece)

        first()
        other.first()

        while (remainSize > 0) {
            read(aBuf)
            other.read(bBuf)

            if (aBuf != bBuf)
                return false
        }

        return true
    }

    var long : Long
        get() = read(Long.SIZE_BYTES).long
        set(value) {
            val buf = ByteBuffer.allocate(Long.SIZE_BYTES)
            buf.putLong(value)
            write(buf)
        }

    var int : Int
        get() = read(Int.SIZE_BYTES).int
        set(value) {
            val buf = ByteBuffer.allocate(Int.SIZE_BYTES)
            buf.putInt(value)
            write(buf)
        }

    var short : Short
        get() = read(Short.SIZE_BYTES).short
        set(value) {
            val buf = ByteBuffer.allocate(Short.SIZE_BYTES)
            buf.putShort(value)
            write(buf)
        }

    var byte : Byte
        get() = read(Byte.SIZE_BYTES).get(0)
        set(value) {
            val buf = ByteBuffer.allocate(Byte.SIZE_BYTES)
            buf.put(value)
            write(buf)
        }

    var bln : Boolean
        get() = byte.bln
        set(value) {
            byte = value.byte
        }

    var array : ByteArray
        get() = read(int).array()
        set(value) {
            int = value.size

            write(ByteBuffer.wrap(value))
        }

    var stream : Stream
        get() = Stream().also {
            it.copyFrom(this, long)
            it.first()
            }
        set(value) {
            long = value.size

            this.copyFrom(value)
        }

    var uuid : UUID
        get() = UUID(long, long)
        set(value) {
            long = value.mostSignificantBits
            long = value.leastSignificantBits
        }

    var time : Date
        get() = Date(long)
        set(value) {
            long = value.time
        }

    var uuids
        get() = int.map { uuid }
        set(value) {
            int = value.size

            value.forEach {
                uuid = it
            }
        }

    var strings
        get() = int.map { string }
        set(value) {
            int = value.size

            value.forEach {
                string = it
            }
        }

    var ints
        get() = int.map { int }
        set(value) {
            int = value.size

            value.forEach {
                int = it
            }
        }

    var longs
        get() = int.map { long }
        set(value) {
            int = value.size

            value.forEach {
                long = it
            }
        }
}