package k.common

import java.io.*
import java.nio.file.*
import kotlin.io.path.isRegularFile

fun forEachFile(dir : String, code : (path : String, stream : InputStream) -> Unit) {
    Files.walk(Path.of(dir)).iterator().forEach {
        val name = it.toString()
        code(name, File(name).inputStream())
    }
}

fun forceDir(dir : String) =
    File(dir).force()

fun File.force() : File =
    this.canonicalFile.absoluteFile.also {
        if (!it.exists())
            it.mkdirs() orThrow "Failed to create $it"
    }

/**
 * Remove file after block
 */
fun useFile(fileName : String, code : (fileName : String) -> Unit) {
    try {
        code(fileName)
    }
    finally {
        if (File(fileName).exists())
            File(fileName).delete()
    }
}

fun <T> useTmpFile(code : (file : File) -> T) : T {
    val file = File("${System.getProperty("java.io.tmpdir")}/$ID")

    try {
        file.createNewFile()
        return code(file)
    }
    finally {
        if (file.exists())
            file.delete()
    }
}

val File.text : String?
    get() = if (this.exists())
        this.readText()
    else
        null

val File.dirs : List<File>
    get() = if (this.exists() && this.isDirectory)
        this.listFiles()?.filter { it.isDirectory }.orEmpty()
    else
        listOf()

val File.files : List<File>
    get() = if (this.exists() && this.isDirectory)
        this.listFiles()?.filter { it.isFile }.orEmpty()
    else
        listOf()

val File.fileTree : List<Path>
    get() = if (exists())
        Files.walk(toPath()).filter { it.isRegularFile() }.toList()
    else
        listOf()