package k.common

import kotlin.math.roundToInt

const val hundredPercent = 100
const val Thousand = 1_000L
const val Million = Thousand * Thousand
const val Billion = Million * Thousand

const val KB = 1024L
const val MB = KB * KB
const val GB = MB * KB
const val TB = GB * KB
const val PB = TB * KB

val Long.amount : String
    get() = amountStr(this)

val Long.isZero : Boolean
    get() = this == 0L

infix fun Long.partOf(value : Long) =
    if (value == 0L)
        0
    else
        ((this.toDouble() / value.toDouble()) * 100).roundToInt()

infix fun Long.percentOf(value : Double) =
    (value * this) / 100

infix fun Long.percentOf(value : Long) =
    (value * this) / 100

val Long.int : Int
    get() = this.toInt()

val Long.negative
    get() = this < 0

val Long.positive
    get() = this > 0

infix fun Long?.default(default : Long) =
    if ((this
         ?: -1) < 0
    )
        default
    else
        this!!

infix fun Long.splitBy(value : Long) =
    (this + value - 1) / value

fun Long.fixRange(min : Long, max : Long) =
    if (this < min)
        min
    else if (this > max)
        max
    else
        this

val Int.bytes
    get() = ByteArray(Int.SIZE_BYTES).also {
        for (i in it.indices) it[i] = (this shr (i * 8)).toByte()
    }

val Int.isNotHundredPercent
    get() = this <= hundredPercent

val Int.isHundredPercent
    get() = this >= hundredPercent

val Long.bytes
    get() = ByteArray(Long.SIZE_BYTES).also { arr ->
        repeat(arr.size) {
            arr[it] = (this shr (it * 8)).toByte()
        }
    }

infix fun Int.percentOf(value : Double) =
    (value * this) / 100

infix fun Int.percentIn(value : Int) =
    (if (value == 0)
        0
    else
        100 * this / value).toDouble()

infix fun Int.percentOf(value : Int) =
    (value * this) / 100

val Int.even
    get() = this % 2 == 0

val Int.odd
    get() = this % 2 == 1

val Int.positive
    get() = this > 0

val Int.negative
    get() = this < 0

infix fun Int?.default(default : Int) =
    if ((this
         ?: -1) < 0
    )
        default
    else
        this!!

val Double.byte
    get() = this.int.toByte()

val Double.int : Int
    get() = this.toInt()

val Double.long : Long
    get() = this.toLong()

infix fun Int.splitBy(value : Int) =
    (this + value - 1) / value

fun Int.fixRange(min : Int, max : Int) =
    if (this < min)
        min
    else if (this > max)
        max
    else
        this

val Int.bln
    get() = this != 0

val Int.long
    get() = toLong()

val Long.bln
    get() = this != 0L

val Byte.bln
    get() = this != 0.toByte()

val Boolean.byte
    get() = (if (this) 1 else 0).toByte()

fun <R> Int.map(transform : (Int) -> R) =
    (0 until this).map { transform(it) }